/***********************************************************************
*
* PATCHDISK - Patches a TI990 tiline disk image file.
*
* Changes:
*      04/20/05   DGP   Original.
*      10/07/18   DGP   Fixed overhead.
*
***********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <errno.h>
#include <ctype.h>

#define NORMAL		0
#define ABORT		16

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#include "simdsk.h"
#include "hexdump.h"
#include "support.h"

/***********************************************************************
* main
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *fd;
   char *fname;
   int dskloc;
   int i;
   int argndx;
   int val;
   int cyls, heads, sectrk, overhead, sectlen;
   int tracklen;
   int head, cyl;
   int sect, offset;
   int verify;
   char buf[290];
   
   fname = NULL;
   cyl = -1;
   head = -1;
   sect = -1;
   offset = -1;
   verify = -1;

   if (argc < 7) goto USAGE;
   for (argndx = 1; argndx < 7; argndx++)
   {
      switch (argndx)
      {
      case 1:
         fname = argv[argndx];
	 break;
      case 2:
         cyl = getnum (argv[argndx]);
	 break;
      case 3:
         head = getnum (argv[argndx]);
	 break;
      case 4:
         sect = getnum (argv[argndx]);
	 break;
      case 5:
	 offset = getnum (argv[argndx]);
	 break;
      case 6:
	 verify = getnum (argv[argndx]);
	 break;
      default:
      USAGE:
	 fprintf (stderr,
 "usage: patchdisk disk.file cylinder head sect offset [verify] [val1 ...]\n");
	 exit (ABORT);
      }
   }
   if (fname == NULL) goto USAGE;
   if (cyl < 0 || head < 0 || sect < 0 || offset < 0 || verify < 0) goto USAGE;

   /*
   ** Open the disk image file
   */

   if ((fd = fopen (fname, "r+b")) == NULL)
   {
      perror ("Can't open disk file");
      exit (ABORT);
   }

   /*
   ** Read the disk geometry
   */

   if ((cyls = dskreadint (fd)) < 0)
   {
      perror ("Can't read disk geometry");
      fclose (fd);
      exit (ABORT);
   }
   if ((heads = dskreadint (fd)) < 0) 
   {
      perror ("Can't read disk geometry");
      fclose (fd);
      exit (ABORT);
   }
   heads &= 0xFFFF;
   if ((dskloc = dskreadint (fd)) < 0)
   {
      perror ("Can't read disk geometry");
      fclose (fd);
      exit (ABORT);
   }
   sectrk = dskloc & 0xFF;
   overhead = (dskloc >> 8) & 0x3FF;
   if ((sectlen = dskreadint (fd)) < 0)
   {
      perror ("Can't read disk geometry");
      fclose (fd);
      exit (ABORT);
   }

   tracklen = sectrk * (sectlen + overhead);

   for (i = 0; i < MAXDISKS; i++)
   {
      if (disks[i].cyls == cyls &&
	  disks[i].heads == heads &&
	  disks[i].sectrk == sectrk &&
	  disks[i].bytsec == sectlen &&
	  disks[i].overhead == overhead)
         break;
   }
   if (i == MAXDISKS)
   {
      printf ("*Unknown Disk model*\n");
   }
	       
   printf ("%s disk geometry:\n", disks[i].model);
   printf ("   cyls      = %d\n", cyls);
   printf ("   heads     = %d\n", heads);
   printf ("   sectrk    = %d\n", sectrk);
   printf ("   sectlen   = %d\n", sectlen);
   printf ("   overhead  = %d\n", overhead);
   printf ("   tracklen  = %d\n", tracklen);
   printf ("   disk size = %d bytes\n", tracklen * heads * cyls);

   if (cyl >= cyls)
   {
      printf ("Invalid disk cylinder specified, abort\n");
      fclose (fd);
      exit (ABORT);
   }
   if (head >= heads)
   {
      printf ("Invalid disk head specified, abort\n");
      fclose (fd);
      exit (ABORT);
   }
   if (sect >= sectrk)
   {
      printf ("Invalid disk sector specified, abort\n");
      fclose (fd);
      exit (ABORT);
   }
   if (offset >= sectlen)
   {
      printf ("Invalid disk offset specified, abort\n");
      fclose (fd);
      exit (ABORT);
   }
   offset &= 0xFFFE;

   /*
   ** Calculate location
   */

   dskloc = (tracklen * head) + (tracklen * heads * cyl) +
	 ((overhead + sectlen) * sect) + DSKOVERHEAD;

#ifdef DEBUG
   printf ("   cyl = %d, head = %d, sector = %d\n",
	    cyl, head, sect);
   printf ("   dskloc = %d\n", dskloc);
   printf ("   offset = %d(%x), verify = %d(%x)\n",
	 offset, offset, verify, verify);
#endif

   /*
   ** Go to the specified location on disk
   */

   if (fseek (fd, dskloc, SEEK_SET) < 0)
   {
      perror ("seek failed");
      fclose (fd);
      exit (ABORT);
   }

   /*
   ** Read sector
   */

   for (i = 0; i < sectlen; i++)
      buf[i] = fgetc (fd) & 0xFF;

   /*
   ** Dump it
   */

   printf ("\ncyl %d head %d sect %d dskloc = %d:\n", cyl, head, sect, dskloc);
   hexdump (stdout, &buf[offset & 0xFFF0], offset & 0xFFF0, 16);

   val = ((buf[offset] & 0xFF) << 8) | (buf[offset+1] & 0xFF);
   if (val != verify)
   {
      printf ("Data does not match verification value\n");
      exit (ABORT);
   }

   i = offset;
   for (; argndx < argc; argndx++)
   {
      val = getnum (argv[argndx]);
      buf[i] = (val >> 8) & 0xFF;
      buf[i+1] = val & 0xFF;
      i += 2;
   }
   printf ("After, dskloc = %d:\n", dskloc);
   hexdump (stdout, &buf[offset & 0xFFF0], offset & 0xFFF0, 16);

   /*
   ** Go to the specified location on disk
   */

   if (fseek (fd, dskloc, SEEK_SET) < 0)
   {
      perror ("seek failed");
      fclose (fd);
      exit (ABORT);
   }

   /*
   ** Write sector
   */

   for (i = 0; i < sectlen; i++)
      if (fputc ((buf[i] & 0xFF), fd) < 0)
      {
	 perror ("Write failed");
	 exit (ABORT);
      }

   fclose (fd);

   return (NORMAL);
}
