/***********************************************************************
*
* SEDIT - Forth screen editor.
*
* Changes:
*      08/25/11   DGP   Original.
*
***********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#define NORMAL		0
#define ABORT		16

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#include "fd800defs.h"

#define SECPSCR 8		/* Sectors per screen */
#define MAXSECTLEN 256		/* Max length of a sector */
#define LINELEN 64		/* Screen line length */
#define DSKOVERHEAD 0

#define EDITOR "/bin/vi"

typedef struct
{
   int cyls;
   int heads;
   int secttrk;
   int sectlen;
   int overhead;
   int tracklen;
   int curcyl;
   int curhead;
   int cursect;
} DiskInfo;

static DiskInfo diskinfo;

/***********************************************************************
* dskopen - Open disk and read geometry.
***********************************************************************/

FILE *
dskopen (char *dskname)
{
   FILE *fd;

#ifdef DEBUGDSK
   fprintf (stderr, "dskopen: dskname = %s\n", dskname);
#endif

   if ((fd = fopen (dskname, "r+b")) == NULL)
   {
      fprintf (stderr, "dskopen: Can't open disk: %s\n", dskname);
      return (NULL);
   }

   /*
   ** Seek to beginning of the disk
   */

   if (fseek (fd, 0, SEEK_SET) < 0)
   {
      fprintf (stderr, "dskopen: seek failed: %s", strerror (errno));
      return (NULL);
   }

   /*
   ** Set the disk geometry
   */

   diskinfo.cyls = FPYTRKDRV;
   diskinfo.heads = 1;
   diskinfo.secttrk = FPYSECTRK;
   diskinfo.overhead = 0;
   diskinfo.sectlen = FPYSECLEN;
   diskinfo.tracklen = diskinfo.secttrk * 
	       (diskinfo.sectlen + diskinfo.overhead);
	       

#ifdef DEBUGDSK
   fprintf (stderr, "Disk geometry:\n");
   fprintf (stderr, "   cyls      = %d\n", diskinfo.cyls);
   fprintf (stderr, "   heads     = %d\n", diskinfo.heads);
   fprintf (stderr, "   secttrk   = %d\n", diskinfo.secttrk);
   fprintf (stderr, "   sectlen   = %d\n", diskinfo.sectlen);
   fprintf (stderr, "   overhead  = %d\n", diskinfo.overhead);
   fprintf (stderr, "   tracklen  = %d\n", diskinfo.tracklen);
   fprintf (stderr, "   disk size = %d bytes\n", diskinfo.tracklen *
	       diskinfo.heads * diskinfo.cyls);
#endif

   return (fd);
}

/***********************************************************************
* dskseek - Seek to specified disk location.
***********************************************************************/

static int
dskseek (FILE *fd, int cyl, int head, int sector)
{
   int dskloc;

   /*
   ** Ensure cylinder is OK
   */

   if ((cyl >= diskinfo.cyls) || (head >= diskinfo.heads) ||
       (sector >= diskinfo.secttrk))
   {
#ifdef DEBUGDSK
      fprintf (stderr, "dskseek: ERROR: cyl = %d, head = %d, sector = %d\n",
	       cyl, head, sector);
#endif
      return (-1);
   }

   /*
   ** Calculate location
   */

   dskloc = (diskinfo.tracklen * head) +
      (diskinfo.tracklen * diskinfo.heads * cyl) +
      ((diskinfo.overhead + diskinfo.sectlen) * sector) +
      DSKOVERHEAD;

#ifdef DEBUGDSK
   fprintf (stderr, "dskseek: cyl = %d, head = %d, sector = %d\n",
	    cyl, head, sector);
   fprintf (stderr, "   dskloc = %d\n", dskloc);
#endif

   /*
   ** Go to the specified location on disk
   */

   if (fseek (fd, dskloc, SEEK_SET) < 0)
   {
      fprintf (stderr, "dskseek: seek failed: %s", strerror (errno));
      return (-1);
   }

   return (0);
}

/***********************************************************************
* Main procedure
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *fd, *tfd;
   int i;
   int cyl, sect, scrnum;
   int firstsec;
   char tmpname[256];
   char tmpbuffer[MAXSECTLEN];
   char screenbuf[SECPSCR][MAXSECTLEN];

   if (argc != 3)
   {
      fprintf (stderr, "usage: sedit forthdisk screen\n");
      exit (ABORT);
   }

   if ((fd = dskopen (argv[1])) == NULL)
      exit (ABORT);

   /*
   ** Convert screen number to disk location.
   */

   scrnum = atoi (argv[2]);
   firstsec = scrnum * SECPSCR;
   cyl = firstsec / diskinfo.secttrk;
   sect = firstsec % diskinfo.secttrk;

   /*
   ** Read in the screen
   */

   for (i = 0; i < SECPSCR; i++)
   {
      if (dskseek (fd, cyl, 0, sect) < 0)
      {
	 fprintf (stderr, "sedit: disk seek error\n");
	 fclose (fd);
	 exit (ABORT);
      }

      if (fread (screenbuf[i], 1, diskinfo.sectlen, fd) != diskinfo.sectlen)
      {
	 fprintf (stderr, "sedit: disk read error\n");
	 fclose (fd);
	 exit (ABORT);
      }

      /* Each sector has two "lines" */
      screenbuf[i][LINELEN-1] = '\n';
      screenbuf[i][LINELEN+LINELEN-1] = '\n';
      screenbuf[i][diskinfo.sectlen] = 0;

      sect++;
      if (sect >= diskinfo.secttrk)
      {
         cyl++;
	 sect = 0;
      }
   }
#ifdef DEBUGDSK
   printf ("SCR # %d\n", scrnum);
   for (i = 0; i < SECPSCR; i ++)
      printf ("%s", screenbuf[i]);
#endif

   /*
   ** Write to tmp file and launch editor.
   */

   sprintf (tmpname, "ED%08d.tmp", getpid());
   if ((tfd = fopen (tmpname, "w")) == NULL)
   {
      fprintf (stderr, "sedit: Can't open temp file: %s\n", tmpname);
      fclose (fd);
      exit (ABORT);
   }
   for (i = 0; i < SECPSCR; i ++)
      fprintf (tfd, "%s", screenbuf[i]);
   fclose (tfd);

   sprintf (tmpbuffer, "%s %s", EDITOR, tmpname);
   system (tmpbuffer);

   /*
   ** Read edited file back into screen buffer.
   */

   if ((tfd = fopen (tmpname, "r")) == NULL)
   {
      fprintf (stderr, "sedit: Can't open temp file: %s\n", tmpname);
      fclose (fd);
      exit (ABORT);
   }

   for (i = 0; i < SECPSCR; i ++)
   {
      char *bp;

      memset (tmpbuffer, ' ', sizeof(tmpbuffer));
      fgets (tmpbuffer, sizeof(tmpbuffer), tfd);
      bp = strchr (tmpbuffer, '\n');
      while ((bp-tmpbuffer) < LINELEN) *bp++ = ' ';
      strncpy (screenbuf[i], tmpbuffer, LINELEN);

      memset (tmpbuffer, ' ', sizeof(tmpbuffer));
      fgets (tmpbuffer, sizeof(tmpbuffer), tfd);
      bp = strchr (tmpbuffer, '\n');
      while ((bp-tmpbuffer) < LINELEN) *bp++ = ' ';
      strncpy (&screenbuf[i][LINELEN], tmpbuffer, LINELEN);
   }
   fclose (tfd);

   /*
   ** Write screen buffer back to disk.
   */

   firstsec = scrnum * SECPSCR;
   cyl = firstsec / diskinfo.secttrk;
   sect = firstsec % diskinfo.secttrk;

   for (i = 0; i < SECPSCR; i++)
   {
      if (dskseek (fd, cyl, 0, sect) < 0)
      {
	 fprintf (stderr, "sedit: disk seek error\n");
	 fclose (fd);
	 exit (ABORT);
      }

      if (fwrite (screenbuf[i], 1, diskinfo.sectlen, fd) != diskinfo.sectlen)
      {
	 fprintf (stderr, "sedit: disk write error\n");
	 fclose (fd);
	 exit (ABORT);
      }

      sect++;
      if (sect >= diskinfo.secttrk)
      {
         cyl++;
	 sect = 0;
      }
   }

   unlink (tmpname);
   fclose (fd);
   return (0);
}
