/***********************************************************************
*
* TAPEIO - TI-990 Tape I/O.
*
* Changes:
*   06/02/16   DGP   Original
*
***********************************************************************/

#include "tapeio.h"


#define MAXTAPE  4

struct device {
   unsigned int status;
   unsigned int wd1;
   unsigned int wd2;
   unsigned int wd3;
   unsigned int count;
   unsigned char *address;
   unsigned int command;
   unsigned int error;
};

static volatile struct device *MTADDR = (struct device *)0xF880;

/*
** Command word control bits
*/

/* Status word 0 - status */

#define OLBIT	0x8000		/* Off Line */
#define BOTBIT	0x4000		/* Beginning Of Tape */
#define EORBIT	0x2000		/* End Of Record */
#define EOFBIT	0x1000		/* End Of File */
#define EOTBIT	0x0800		/* End Of Tape */
#define WPBIT	0x0400		/* Write Protect */
#define RWBIT	0x0200		/* Tape Rewinding */
#define TOBIT	0x0100		/* Command Time Out */

/* Status word 7  - error */

#define IDLEBIT	0x8000		/* Idle */
#define COMPBIT	0x4000		/* Complete */
#define ERRBIT	0x2000		/* Error */
#define INTBIT	0x1000		/* Interrupt Enable */
#define sp0BIT	0x0800		/* spare */
#define sp1BIT	0x0400		/* spare */
#define PEFBIT	0x0200		/* PE Format */
#define ABNBIT	0x0100		/* Abnormal Completion */
#define IPEBIT	0x0080		/* Interface Parity error */
#define ECEBIT	0x0040		/* Error Correction enabled */
#define HWEBIT	0x0020		/* Hard Error */
#define TMRBIT	0x0010		/* Tiline memory read error */
#define TTEBIT	0x0008		/* Tiline Timing error */
#define TTOBIT	0x0004		/* Tiline Timout error */
#define FMEBIT	0x0002		/* Format error */
#define TPEBIT	0x0001		/* Tape error check status word 0 */

#define HARDERR	0x00BE		/* IPE|HWE|TMR|TTE|TTO|FMT */

#define MTNOP	0		/* No op command */
#define MTWEOF	2		/* Write EOF */
#define MTBSR	3		/* Backspace record */
#define MTREAD	4		/* Tape read */
#define MTFSR	5		/* Forward skip record */
#define MTWRITE	6		/* Tape write */
#define MTREW	10		/* Rewind */
#define MTRWUNL	11		/* Rewind/Unload */

static int tpunit[MAXTAPE] = { 8, 4, 2, 1 };

static int
tapeio (int unit, int cmd, char *buff, int len)
{
   int done;

#ifdef DEBUGTAPE
   printf ("tapeio: unit = %d, cmd = %d, buff = >%04X, len = %d\n",
	   unit, cmd, buff, len);
#endif

   MTADDR->status = 0;
   MTADDR->wd1 = 0;
   MTADDR->wd2 = 0;
   MTADDR->wd3 = 0;
   MTADDR->count = len;
   MTADDR->address = (unsigned char *)buff;
   MTADDR->command = tpunit[unit] << 12 | cmd << 8;
   MTADDR->error = 0;

   while ((MTADDR->error & IDLEBIT) == 0) ;

#ifdef DEBUGTAPE
   printf (" complete: error = >%04X\n", MTADDR->error & HARDERR);
   printf ("   status = >%04X\n", MTADDR->status);
   printf ("   count  = >%04X\n", MTADDR->count);
   printf ("   error  = >%04X\n", MTADDR->error);
#endif
   return (MTADDR->error & HARDERR);
}

int
taperead (int unit, char *buff, int len)
{
   int ret;

   ret = tapeio (unit, MTREAD, buff, len);

   if (MTADDR->status & EOFBIT)
      return (0);

   if (ret != 0)
      return (-1);

   return (len - MTADDR->count);
}

int
tapewrite (int unit, char *buff, int len)
{
   int ret;

   ret = tapeio (unit, MTWRITE, buff, len);

   if (MTADDR->status & EOTBIT)
      return (0);

   if (ret != 0)
      return (-1);

   return (len);
}

int
tapeweof (int unit)
{
   int ret;

   if ((ret = tapeio (unit, MTWEOF, NULL, 0)) != 0)
      return (-1);

   return (0);
}

int
taperewind (int unit)
{
   int ret;

   if ((ret = tapeio (unit, MTREW, NULL, 0)) != 0)
      return (-1);

   return (0);
}
