/***********************************************************************
*
* DCOPY - Copy TI-990 disk image to/from a "real" disk.
*
* Changes:
*   01/26/16   DGP   Original
*   04/21/16   DGP   Made symetric operation.
*   10/07/18   DGP   Fixed overhead.
*
***********************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "hexdump.h"
#include "support.h"
#include "simdsk.h"

typedef struct {
   char model[256];
   int tracklen;
   int cyls;
   int seccyl;
   int heads;
   int sectrk;
   int overhead;
   int sectlen;
} Geometry_t;

static int verbose;

/***********************************************************************
* Lookup the input disk geometry
***********************************************************************/

static int
lookupgeometry (char *modelname, Geometry_t *geometry)
{
   int i;

   for (i = 0; i < MAXDISKS; i++)
   {
      if (!strcmp (disks[i].model, modelname))
         break;
   }
   if (i == MAXDISKS)
   {
      fprintf (stderr, "dcopy: Unknown Disk model: %s\n", modelname);
      return (-1);
   }
   geometry->cyls = disks[i].cyls;
   geometry->heads = disks[i].heads;
   geometry->sectrk = disks[i].sectrk;
   geometry->sectlen = disks[i].bytsec;
   geometry->overhead = disks[i].overhead;

   geometry->tracklen = geometry->sectrk *
	    (geometry->sectlen + geometry->overhead);
   geometry->seccyl = geometry->heads * geometry->sectrk;
   strcpy (geometry->model, disks[i].model);
	       
   if (verbose)
   {
      printf ("%s lookup disk geometry:\n", geometry->model);
      printf ("   cyls      = %d\n", geometry->cyls);
      printf ("   heads     = %d\n", geometry->heads);
      printf ("   sectrk    = %d\n", geometry->sectrk);
      printf ("   sectlen   = %d\n", geometry->sectlen);
      printf ("   overhead  = %d\n", geometry->overhead);
      printf ("   tracklen  = %d\n", geometry->tracklen);
      printf ("   disk size = %d bytes, %d sectors\n",
	       geometry->tracklen * geometry->heads * geometry->cyls,
	       geometry->cyls * geometry->seccyl);
   }
   return (0);
}

/***********************************************************************
* Read the input disk geometry
***********************************************************************/

static int
readgeometry (FILE *ifd, Geometry_t *geometry)
{
   int i;

   if ((geometry->cyls = dskreadint (ifd)) < 0)
      goto DISKERROR;
   if ((geometry->heads = dskreadint (ifd)) < 0) 
      goto DISKERROR;
   geometry->heads &= 0xFFFF;
   if ((i = dskreadint (ifd)) < 0)
      goto DISKERROR;
   geometry->sectrk = i & 0xFF;
   geometry->overhead = (i >> 8) & 0x3FF;
   if ((geometry->sectlen = dskreadint (ifd)) < 0)
      goto DISKERROR;

   geometry->tracklen = geometry->sectrk *
	    (geometry->sectlen + geometry->overhead);
   geometry->seccyl = geometry->heads * geometry->sectrk;

   for (i = 0; i < MAXDISKS; i++)
   {
      if (disks[i].cyls == geometry->cyls &&
	  disks[i].heads == geometry->heads &&
	  disks[i].sectrk == geometry->sectrk &&
	  disks[i].bytsec == geometry->sectlen &&
	  disks[i].overhead == geometry->overhead)
	 break;
   }
   if (i == MAXDISKS)
   {
      fprintf (stderr, "dcopy: Unknown Disk model\n");
      return (-1);
   }
   strcpy (geometry->model, disks[i].model);
	       
   if (verbose)
   {
      printf ("%s read disk geometry:\n", geometry->model);
      printf ("   cyls      = %d\n", geometry->cyls);
      printf ("   heads     = %d\n", geometry->heads);
      printf ("   sectrk    = %d\n", geometry->sectrk);
      printf ("   sectlen   = %d\n", geometry->sectlen);
      printf ("   overhead  = %d\n", geometry->overhead);
      printf ("   tracklen  = %d\n", geometry->tracklen);
      printf ("   disk size = %d bytes, %d sectors\n",
	       geometry->tracklen * geometry->heads * geometry->cyls,
	       geometry->cyls * geometry->seccyl);
   }
   return (0);

DISKERROR:
   perror ("dcopy: Can't read disk geometry");
   return (-1);
}

/***********************************************************************
* Write the output disk geometry
***********************************************************************/

static int
writegeometry (FILE *ofd, Geometry_t *geometry)
{
   if (verbose)
   {
      printf ("%s write disk geometry:\n", geometry->model);
      printf ("   cyls      = %d\n", geometry->cyls);
      printf ("   heads     = %d\n", geometry->heads);
      printf ("   sectrk    = %d\n", geometry->sectrk);
      printf ("   sectlen   = %d\n", geometry->sectlen);
      printf ("   overhead  = %d\n", geometry->overhead);
      printf ("   tracklen  = %d\n", geometry->tracklen);
      printf ("   disk size = %d bytes, %d sectors\n",
	       geometry->tracklen * geometry->heads * geometry->cyls,
	       geometry->cyls * geometry->seccyl);
   }

   if (dskwriteint (ofd, geometry->cyls) < 0)
      goto DISKERROR;
   if (dskwriteint (ofd, geometry->heads) < 0)
      goto DISKERROR;
   if (dskwriteint (ofd, (geometry->overhead << 8) | geometry->sectrk) < 0)
      goto DISKERROR;
   if (dskwriteint (ofd, geometry->sectlen) < 0)
      goto DISKERROR;

   return (0);

DISKERROR:
   perror ("dcopy: Can't write disk geometry");
   return (-1);
}

/***********************************************************************
* main
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *ifd;
   FILE *ofd;
   char *indisk;
   char *outdisk;
   char *modelname;
   char *bp;
   char *mem;
   size_t size;
   int status;
   int indiskreal, outdiskreal;
   int i, j, k;
   Geometry_t geometry;

   /*
   ** Scan off args 
   */

   indisk = NULL;
   outdisk = NULL;
   modelname = NULL;
   mem = NULL;
   verbose = FALSE;
   indiskreal = FALSE;
   outdiskreal = FALSE;
   status = 0;

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
         for (bp++; *bp; bp++) switch (*bp)
         {
	 case 'm':
	    i++;
	    modelname = argv[i];
	    break;

	 case 'v':
            verbose = TRUE;
	    break;

         default:
      USAGE:
	    printf ("usage: dcopy [-options] in.disk out.disk\n");
            printf (" options:\n");
            printf ("    -m MODEL     - Model name\n");
            printf ("    -v           - Verbose mode\n");
	    exit (1);
         }
      }
      else if (!indisk)
      {
        indisk = argv[i];
      }
      else if (!outdisk)
      {
        outdisk = argv[i];
      }
      else
      {
         goto USAGE;
      }

   }
   if (!indisk) goto USAGE;
   if (!outdisk) goto USAGE;

   if (!strncmp (indisk, "/dev/", 5))
   {
      indiskreal = TRUE;
      if (!modelname)
      {
         fprintf (stderr, "dcopy: Model name required for real input disk\n");
	 goto USAGE;
      }
   }
   if (!strncmp (outdisk, "/dev/", 5))
   {
      outdiskreal = TRUE;
   }

   if (indiskreal && outdiskreal)
   {
      fprintf (stderr, "dcopy: Both disks can't be real\n");
      exit (1);
   }

   if ((ifd = fopen (indisk, "rb")) == NULL)
   {
      perror ("dcopy: Can't open input disk");
      exit (1);
   }

   if (lseek (fileno(ifd), 0L, SEEK_SET) < 0)
   {
      perror ("dcopy: Can't seek on input disk");
      fclose (ifd);
      exit (1);
   }

   if (indiskreal)
   {
      if (lookupgeometry (modelname, &geometry) < 0)
      {
	 fclose (ifd);
         exit (1);
      }
   }
   else
   {
      if (readgeometry (ifd, &geometry) < 0)
      {
	 fclose (ifd);
         exit (1);
      }
   }

   if ((ofd = fopen (outdisk, "wb")) == NULL)
   {
      perror ("dcopy: Can't open output disk");
      fclose (ifd);
      exit (1);
   }

   if (lseek (fileno(ofd), 0L, SEEK_SET) < 0)
   {
      perror ("dcopy: Can't seek on output disk");
      status = 1;
      goto DIE;
   }

   if (!outdiskreal)
   {
      if (writegeometry (ofd, &geometry) < 0)
      {
	 status = 1;
	 goto DIE;
      }
   }

   if ((mem = malloc (512)) == NULL)
   {
      fprintf (stderr, "dcopy: Unable to allocate buffer\n");
      status = 1;
      goto DIE;
   }

   k = 0;
   for (i = 0; i < geometry.cyls; i++)
   {
      /*
      if (verbose)
	 printf ("Copy cylinder %d\n", k++);
      */
      for (j = 0; j < geometry.seccyl; j++)
      {
	 if (fread (mem, 1, geometry.sectlen, ifd) != geometry.sectlen)
	 {
	    perror ("dcopy: Can't read input disk");
	    status = 1;
	    goto DIE;
	 }
	 if (fwrite (mem, 1, geometry.sectlen, ofd) != geometry.sectlen)
	 {
	    perror ("dcopy: Can't write output disk");
	    status = 1;
	    goto DIE;
	 }
	 if (geometry.overhead)
	 {
	    if (fread (mem, 1, geometry.overhead, ifd) != geometry.overhead)
	    {
	       perror ("dcopy: Can't read input disk");
	       status = 1;
	       goto DIE;
	    }
	    if (fwrite (mem, 1, geometry.overhead, ofd) != geometry.overhead)
	    {
	       perror ("dcopy: Can't write output disk");
	       status = 1;
	       goto DIE;
	    }
	 }
      }
   }

   if (verbose)
      printf ("Sync disk\n");
   fsync (fileno(ofd));

DIE:
   if (mem) free (mem);
   fclose (ifd);
   fclose (ofd);
   
   return (status);
}
