/***********************************************************************
*
* DISKIO - TI-990 Disk I/O.
*
* Changes:
*   06/02/16   DGP   Original
*
***********************************************************************/

#include "diskio.h"

#define MAXDISK  4

struct device {
   unsigned int status;
   unsigned int command;
   unsigned int sect;
   unsigned int cyl;
   unsigned int count;
   unsigned char *address;
   unsigned int unit;
   unsigned int error;
};

static volatile struct device *DSKADDR = (struct device *)0xF800;

/*
** Command word control bits
*/

/* Status word 0 - status */

#define OLBIT	0x8000		/* Off Line */
#define NRBIT	0x4000		/* Not Ready */
#define WPBIT	0x2000		/* Write Protect */
#define USBIT	0x1000		/* UnSafe */
#define ECBIT	0x0800		/* End of Cylinder */
#define SIBIT	0x0400		/* Seek Incomplete */
#define OSABIT	0x0200		/* Offset Active */
#define PCBIT	0x0100		/* Unsafe reason */

/* Status word 7  - error */

#define IDLEBIT	0x8000		/* Idle */
#define COMPBIT	0x4000		/* Complete */
#define ERRBIT	0x2000		/* Error */
#define INTBIT	0x1000		/* Interrupt enable */
#define LOCKBIT	0x0800		/* Lockout */
#define RETBIT	0x0400		/* Retry */
#define ECCBIT	0x0200		/* ECC corrected */
#define ABNBIT	0x0100		/* Abnormal Completion */
#define MEBIT	0x0080		/* Memory Error */
#define DEBIT	0x0040		/* Data Error */
#define TTBIT	0x0020		/* Tiline Timeout */
#define IEBIT	0x0010		/* ID Error */
#define REBIT	0x0008		/* Rate Error */
#define CTBIT	0x0004		/* Command Timer */
#define SEBIT	0x0002		/* Search Error */
#define UEBIT	0x0001		/* Unit Error */

/* Hard error mask - 0x21FF */
#define HARDERR (ERRBIT|ABNBIT|MEBIT|DEBIT|TTBIT|IEBIT|REBIT|CTBIT|SEBIT|UEBIT)

/* Disk commands */

#define STOREREG  0		/* Store registers */
#define WRITEFMT  1		/* Write format */
#define READDATA  2		/* Read data */
#define WRITEDATA 3		/* Write data */
#define READUFMT  4		/* Read unformated */
#define WRITEUFMT 5		/* Write unformated */
#define DISKSEEK  6		/* Disk Seek */
#define RESTORE   7		/* Restore */

static int dskunit[MAXDISK] = { 8, 4, 2, 1 };

static int
diskio (int unit, int cmd, int cyl, int head, int sect, char *buff, int len)
{
#ifdef DEBUGDISK
   printf ("diskio: unit = %d, cmd = %d, buff = >%04X, len = %d\n",
	   unit, cmd, buff, len);
   printf ("   cyl = %d, head = %d, sect = %d\n", cyl, head, sect);
#endif

   DSKADDR->status = 0;
   DSKADDR->command = (cmd << 8) | head;
   DSKADDR->sect = 0x100 | sect;
   DSKADDR->cyl = cyl;
   DSKADDR->count = len;
   DSKADDR->address = (unsigned char *)buff;
   DSKADDR->unit = dskunit[unit] << 8;
   DSKADDR->error = 0;

WAIT:
   while ((DSKADDR->error & IDLEBIT) == 0) ;
   if (cmd == DISKSEEK)
   {
      if ((DSKADDR->status & (dskunit[unit] << 4)) == 0)
         goto WAIT;
   }

   if (DSKADDR->error & HARDERR)
   {
      printf (
	   "Disk error: unit = %d, cmd = %d, cyl = %d, head = %d, sec = %d\n",
	   unit, cmd, cyl, head, sect);
      printf ("   status = >%04X\n", DSKADDR->status);
      printf ("   error  = >%04X\n", DSKADDR->error);
   }
   return (DSKADDR->error & HARDERR);
}

int
diskread (int unit, int cyl, int head, int sect, char *buff, int len)
{
   if (diskio (unit, DISKSEEK, cyl, head, sect, NULL, 0) != 0)
      return (-1);
   if (diskio (unit, READDATA, cyl, head, sect, buff, len) != 0)
      return (-1);

   return (len);
}

int
diskwrite (int unit, int cyl, int head, int sect, char *buff, int len)
{
   if (diskio (unit, DISKSEEK, cyl, head, sect, NULL, 0) != 0)
      return (-1);
   if (diskio (unit, WRITEDATA, cyl, head, sect, buff, len) != 0)
      return (-1);

   return (len);
}

int
diskinit (int unit)
{
   if (diskio (unit, RESTORE, 0, 0, 0, NULL, 0) != 0)
      return (-1);

   return (0);
}

int
diskgeometry (int unit, int *cyl, int *head, int *sect, int *len)
{
   unsigned int regs[3];
   unsigned int fmt[3];

   if (diskio (unit, STOREREG, 0, 0, 0, (char *)&regs, 6) != 0)
      return (-1);
   if (diskio (unit, READUFMT, 0, 0, 0, (char *)&fmt, 6) != 0)
      return (-1);

#ifdef DEBUGDISK
   printf ("diskgeometry: wrd/trk = %d, sect/trk = %d, overhead = %d\n",
	   regs[0], regs[1] >> 8, regs[1] & 0xFF);
   printf ("   heads = %d, cyls = %d, sectlen = %d\n", 
	   regs[2] >> 11, regs[2] & 0x7FF, fmt[2] << 1);
#endif

   *cyl = regs[2] & 0x7FF;
   *head = (regs[2] >> 11) & 0x1F;
   *sect = (regs[1] >> 8) & 0xFF;
   *len = fmt[2] << 1;

   return (0);
}
