/***********************************************************************
*
* disops.c - Simulate instruction operations for the TI 990 computer.
*
* Changes:
*   05/05/05   DGP   Original. Hacked from simops.c
*   01/10/07   DGP   Added function branch table.
*   01/16/07   DGP   Added DSEG support.
*   10/31/07   DGP   Generate AORG when images load above zero.
*   12/04/07   DGP   Add DSEG disassembly.
*   11/10/08   DGP   Added /12 support.
*   08/09/11   DGP   Added CPU model support.
*   08/17/11   DGP   Allow multiple CSEGs.
*   10/10/13   DGP   Fixed Inst operand decode if it's an org.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>

#include "utildef.h"

extern int pcreg;	/* The program PC */
extern int pgmlen;	/* The program length */
extern int dseglen;	/* The program DSEG length */
extern int symbolcount;
extern int model;

extern int cseglen[MAXCSEGS];	/* The program CSEG length */

extern char idtbuf[IDTSIZE+2];
extern char csegname[MAXCSEGS][IDTSIZE+2];

extern uint8 memory[MEMSIZE];
extern Memory memctl[MEMSIZE];
extern uint8 dsegmem[MEMSIZE];
extern Memory dsegctl[MEMSIZE];
extern uint8 csegmem[MAXCSEGS][MEMSIZE];
extern Memory csegctl[MAXCSEGS][MEMSIZE];
extern SymNode *symbols[MAXSYMBOLS];

static int srcaddre;
static int srcaddro;
static int dstaddre;
static int dstaddro;
static int segtype;
static int csegndx;

static char outbuf[82];

/*
** Opcode tables.
*/

static char *ops1op[] =
{
   "",     "",     "",     "",     "SZC",  "SZCB", "S",    "SB",
   "C",    "CB",   "A",    "AB",   "MOV",  "MOVB", "SOC",  "SOCB"
};

static char *ops2op[] =
{
   "JMP",  "JLT",  "JLE",  "JEQ",  "JHE",  "JGT",  "JNE",  "JNC",
   "JOC",  "JNO",  "JL",   "JH",   "JOP",  "SBO",  "SBZ",  "TB"
};

static char *ops3op[] =
{
   "COC",  "CZC",  "XOR",  "XOP"
};

static char *ops4op[] =
{
   "LDCR", "STCR"
};

static char *ops5op[] =
{
   "SRA",  "SRL",  "SLA",  "SRC"
};

static char *ops6op[] =
{
   "BLWP", "B",    "X",    "CLR",  "NEG",  "INV",  "INC",  "INCT",
   "DEC",  "DECT", "BL",   "SWPB", "SETO", "ABS",  "LDS",  "LDD",
   "",     "",     "",     ""
};
static char *ops6fop[] = 
{
   "",     "AR",   "CIR",  "SR",   "MR",   "DR",   "LR",   "STR",
   "",     "AD",   "CID",  "SD",   "MD",   "DD",   "LD",   "STD"
};
static char *ops6eop[] = 
{
   "",     "BIND", "DIVS", "MPYS"
};

static char *ops7op[] =
{
   "",     "",     "",     "",     "IDLE", "",     "RSET", "",
   "RTWP", "",     "CKON", "",     "CKOF", "",     "LREX"
};
static char *ops7fop[] = 
{
   "CRI",  "CDI",  "NEGR", "NEGD", "CRE",  "CDE",  "CER",  "CED",
   "",     "",     "",     "",     "",     "",     "XIT",  "XIT"
};
static char *ops7eop[] = 
{
   "",     "",     "",     "",     "",     "",     "",     "",
   "",     "",     "",     "",     "",     "EMD",  "EINT", "DINT"
};

static char *ops8op[] =
{
   "LI",   "",     "AI",   "",     "ANDI", "",     "ORI",  "",
   "CI",   "",     "STWP", "",     "STST", "",     "LWPI"
};

static char *ops9op[] =
{
   "MPY",  "DIV"
};

static char *ops11op[] =
{
   "RTO",  "LTO",  "CNTO", "",     "",     "BDC",  "DBC",  "SWPM",
   "XORM", "ORM",  "ANDM", "SM",   "AM"
};

static char *ops12aop[] =
{
   "",     "SNEB", "CRC",  "TS"
};
static char *ops12bop[] =
{
   "CS",   "SEQB", "MOVS", "",     "",     "",     "",     "",
   "MVSR", "MVSK", "POPS", "PSHS"
};

static char *ops13op[] =
{
   "SRAM", "SLAM"
};

static char *ops14op[] =
{
   "TMB",  "TCMB", "TSMB"
};

static char *ops16op[] =
{
   "",     "INSF", "XV",   "XF"
};

static char *ops17op[] =
{
   "SRJ",  "ARJ"
};

static char *ops18op[] =
{
   "STPC", "",     "",     "",     "LIM",  "LST",  "LWP",  "LCS"
};

static char *ops20op[] =
{
   "",     "SLSL", "SLSP"
};
static char *ops20cc[] =
{
   "EQ",   "NE",   "HE",   "L",    "GE",   "LT",   "LE",   "H",
   "LTE",  "GT"
};

/*
** Function definitions and branch table
*/

#define MAX_INST_TYPES 21

static int proc1op  (uint16 inst);
static int proc2op  (uint16 inst);
static int proc3op  (uint16 inst);
static int proc4op  (uint16 inst);
static int proc5op  (uint16 inst);
static int proc6op  (uint16 inst);
static int proc7op  (uint16 inst);
static int proc8op  (uint16 inst);
static int proc9op  (uint16 inst);
static int proc10op (uint16 inst);
static int proc11op (uint16 inst);
static int proc12op (uint16 inst);
static int proc13op (uint16 inst);
static int proc14op (uint16 inst);
static int proc15op (uint16 inst);
static int proc16op (uint16 inst);
static int proc17op (uint16 inst);
static int proc18op (uint16 inst);
static int proc19op (uint16 inst);
static int proc20op (uint16 inst);
static int proc21op (uint16 inst);

typedef struct {
   int (*proc) (uint16 inst);
} Inst_Proc;

Inst_Proc inst_proc[MAX_INST_TYPES] = {
   proc1op,  proc2op,  proc3op,  proc4op,  proc5op,  proc6op,  proc7op,
   proc8op,  proc9op,  proc10op, proc11op, proc12op, proc13op, proc14op,
   proc15op, proc16op, proc17op, proc18op, proc19op, proc20op, proc21op
};

/***********************************************************************
* printvals - Print symbols whose value equal val.
***********************************************************************/

static void
printvals (int val, int inc)
{
   SymNode *s;
   int i;

   for (i = 0; i < symbolcount; i++)
   {
      int discond, ndxcond;

      s = symbols[i];
      discond = segtype ? (s->flags & segtype) : !(s->flags & (CSEG|DSEG));
      ndxcond = s->cmnndx == csegndx;
      if (!(s->flags & EXTERNAL) && (s->flags & RELOCATABLE) && discond &&
	   ndxcond && (s->value == val))
      {
	 printf ("%s\tEQU\t$", s->symbol);
	 if (inc > 0)
	    printf ("+%d", inc);
	 else if (inc < 0)
	    printf ("%d", inc);
         printf ("\n");
      }
   }
}

/***********************************************************************
* decodeinst - Decode instructions.
***********************************************************************/

static int
decodeinst (uint16 inst)
{
   if (inst & 0xF000)
   {
      if (inst & 0xC000) return TYPE_1;
      if ((inst & 0xF800) == 0x3000) return TYPE_4;
      if ((inst & 0xF000) == 0x3000) return TYPE_9;
      if ((inst & 0xF000) == 0x2000) return TYPE_3;
      return TYPE_2;
   }
   else if (inst & 0x0F00)
   {
      if ((inst & 0x0F00) == 0x0F00) return TYPE_6;
      else if ((inst & 0x0F00) == 0xE00)
      {
	 if ((inst & 0x0FF0) == 0x0E00) return TYPE_15;
	 if ((inst & 0x0FC0) == 0x0E00) return TYPE_12;
	 return TYPE_6;
      }
      else if ((inst & 0x0F00) == 0x0D00) return TYPE_6;
      else if ((inst & 0x0F00) == 0x0C00)
      {
	 if ((inst & 0x0FFF) == 0x0C0F) return TYPE_7;
	 if ((inst & 0x0FFF) == 0x0C0E) return TYPE_7;
	 if ((inst & 0x0FFF) == 0x0C08) return TYPE_11;
	 if ((inst & 0x0FFE) == 0x0C0C) return TYPE_17;
	 if ((inst & 0x0FF8) == 0x0C08) return TYPE_14;
	 if ((inst & 0x0FF0) == 0x0C00) return TYPE_7;
	 if ((inst & 0x0FC0) == 0x0C00) return TYPE_16;
	 return TYPE_6;
      }
      else if ((inst & 0x0C00) == 0x0800) return TYPE_5;
      else if ((inst & 0x0C00) == 0x0400) return TYPE_6;
      else if ((inst & 0x0F00) == 0x0300)
      {
	 if (inst == 0x0300) return TYPE_8;
	 if ((inst & 0x0FF0) == 0x03F0) return TYPE_21;
	 if ((inst & 0x0FE0) == 0x0320) return TYPE_10;
	 return TYPE_7;
      }
      else if ((inst & 0x0F00) == 0x0200) return TYPE_8;
      else if ((inst & 0x0F00) == 0x0100) return TYPE_6;
   }
   else if (inst & 0x00F0)
   {
      if ((inst & 0x00F0) == 0x00B0) return TYPE_8;
      else if ((inst & 0x00F0) == 0x0070) return TYPE_18;
      else if ((inst & 0x0040) == 0x0040) return TYPE_12;
      else if ((inst & 0x00F0) == 0x0020)
      {
	 if ((inst & 0x00FF) == 0x0021) return TYPE_20;
	 if ((inst & 0x00FF) == 0x0022) return TYPE_20;
	 if ((inst & 0x00FF) == 0x002B) return TYPE_19;
	 if ((inst & 0x00FC) == 0x002C) return TYPE_7;
	 if ((inst & 0x00FC) == 0x0028) return TYPE_11;
	 return TYPE_11;
      }
      else if ((inst & 0x00F0) == 0x0010)
      {
	 if ((inst & 0x00FE) == 0x001E) return TYPE_11;
	 return TYPE_13;
      }
      return TYPE_18;
   }
   return (TYPE_ILL);
}


/***********************************************************************
* getctl - Get mem control address.
***********************************************************************/

static Memory *
getctl (void)
{
   if (segtype == DSEG)
      return (&dsegctl[pcreg]);
   if (segtype == CSEG)
      return (&csegctl[csegndx][pcreg]);
   return (&memctl[pcreg]);
}

/***********************************************************************
* getinst - Get an Instruction.
***********************************************************************/

static uint16
getinst (void)
{
   uint16 inst;

   if (segtype == DSEG)
      inst = GETDSEGMEM(pcreg);
   else if (segtype == CSEG)
      inst = GETCSEGMEM(csegndx, pcreg);
   else
      inst = GETMEM (pcreg);
   pcreg += 2;

   return (inst);
}

/***********************************************************************
* genoperand - Process general operand
***********************************************************************/

static int
genoperand (uint16 sr, uint16 st, int dst)
{
   Memory *lclctl;
   SymNode *s;
   int dsegdata;
   uint16 sa;
   char tbuf[40];


   switch (st)
   {
   case 0:

      sprintf (tbuf, "R%d", sr);
      break;

   case 1:

      sprintf (tbuf, "*R%d", sr);
      break;

   case 2:

      lclctl = getctl();
      if (lclctl->tag == ABSORG_TAG || lclctl->tag == RELORG_TAG)
         return (-1);

      s = NULL;
      dsegdata = FALSE;

      sa = getinst();

      if (lclctl->tag == EXTNDX_TAG)
         s = extlookup (sa, TRUE);
      else if (!lclctl->external && lclctl->relocatable)
	 s = vallookup (sa, FALSE, lclctl->tag == DSEGDATA_TAG ? TRUE : FALSE,
	 		csegndx);
      else if (lclctl->external)
         s = extlookup (sa, FALSE);

      if (s && !(s->flags & DSEG))
      {
	 sprintf (tbuf, "@%s", s->symbol);
	 if (lclctl->value)
	    sprintf (&tbuf[strlen(tbuf)], "+%d", lclctl->value);
      }
      else if (!dsegdata)
      {
	 int16 ssa = sa;
	 sprintf (tbuf, "@%d", ssa);
      }

      if (sr)
	 sprintf (&tbuf[strlen(tbuf)], "(R%d)", sr);

      if ((s = vallookup (pcreg-2, FALSE, segtype, csegndx)) != NULL)
      {
	 srcaddre = pcreg - 2;
      }
      if ((s = vallookup (pcreg-1, FALSE, segtype, csegndx)) != NULL)
      {
	 srcaddro = pcreg - 1;
      }
      break;

   case 3:

      sprintf (tbuf, "*R%d+", sr);
      break;
   }

   if (dst) strcat (tbuf, ",");

   strcat (outbuf, tbuf);
   return (0);
}

/***********************************************************************
* proc1op - Process type 1 operations.
***********************************************************************/

static int
proc1op (uint16 inst)
{
   uint16 st;
   uint16 sr;
   uint16 dt;
   uint16 dr;
   uint16 op;

   sr =  inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   dr = (inst & 0x03C0) >> 6;
   dt = (inst & 0x0C00) >> 10;
   op = (inst & 0xF000) >> 12;

   switch (op)
   {
   case 4:  /* SZC  */
   case 5:  /* SZCB */
   case 6:  /* S    */
   case 7:  /* SB   */
   case 8:  /* C    */
   case 9:  /* CB   */
   case 10: /* A    */
   case 11: /* AB   */
   case 12: /* MOV  */
   case 13: /* MOVB */
   case 14: /* SOC  */
   case 15: /* SOCB */
      break;
   default:
      return (-1);
   }

   sprintf (outbuf, "\t%s\t", ops1op[op]);
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   if (genoperand (dr, dt, FALSE) < 0) return (-1);

   return (0);
}

/***********************************************************************
* proc2op - Process type 2 operations.
***********************************************************************/

static int
proc2op (uint16 inst)
{
   SymNode *s;
   int16 wpc;
   uint16 op;
   int8 disp;

   disp = inst & 0x00FF;
   op = (inst & 0x0F00) >> 8;

   switch (op)
   {
   case 0:  /* JMP */
   case 1:  /* JLT */
   case 2:  /* JLE */
   case 3:  /* JEQ */
   case 4:  /* JHE */
   case 5:  /* JGT */
   case 6:  /* JNE */
   case 7:  /* JNC */
   case 8:  /* JOC */
   case 9:  /* JNO */
   case 10: /* JL  */
   case 11: /* JH  */
   case 12: /* JOP */
      break;
   case 13: /* SBO */
   case 14: /* SBZ */
   case 15: /* TB  */
      sprintf (outbuf, "\t%s\t%d", ops2op[op], disp);
      return (0);
   default:
      return (-1);
   }

   if (inst == 0x1000)
   {
      strcpy (outbuf, "\tNOP\t");
   }
   else
   {
      uint16 upc;

      wpc = (pcreg >> 1) & 0x7FFF;
      wpc += disp;
      wpc = (wpc << 1) & 0xFFFE;
      upc = wpc;

      sprintf (outbuf, "\t%s\t", ops2op[op]);
      if ((uint8)disp == 0xFF)
      {
         strcat (outbuf, "$");
      }
      else
      {
	 if ((s = vallookup (upc, FALSE, segtype, csegndx)) != NULL)
	    sprintf (&outbuf[strlen(outbuf)], "%s\t", s->symbol);
	 else if (disp < 0)
	    sprintf (&outbuf[strlen(outbuf)], "$-%d\t", (-disp)*2-2);
	 else
	    sprintf (&outbuf[strlen(outbuf)], "$+%d\t", disp*2+2);

	 sprintf (&outbuf[strlen(outbuf)], ">%04X\t>%02X", upc, disp & 0xFF);
      }
   }

   return (0);
}

/***********************************************************************
* proc3op - Process type 3 operations.
***********************************************************************/

static int
proc3op (uint16 inst)
{
   uint16 st;
   uint16 sr;
   uint16 reg;
   uint16 op;
   char chop[16];

   sr =  inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   reg = (inst & 0x03C0) >> 6;
   op = (inst & 0x1C00) >> 10;

   switch (op)
   {
   case 0: /* COC */
   case 1: /* CZC */
   case 2: /* XOR */
      sprintf (chop, "R%d", reg);
      break;
   case 3: /* XOP */
      if (reg == 15)
      {
	 sprintf (outbuf, "\t%s\t", "SVC");
	 if (genoperand (sr, st, FALSE) < 0) return (-1);
	 return (0);
      }
      sprintf (chop, "%d", reg);
      break;
   default:
      return (-1);
   }

   sprintf (outbuf, "\t%s\t", ops3op[op]);
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   strcat (outbuf, chop);

   return (0);
}

/***********************************************************************
* proc4op - Process type 4 operations.
***********************************************************************/

static int
proc4op (uint16 inst)
{
   uint16 sr;
   uint16 st;
   uint16 cnt;
   uint16 op;

   sr = inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   cnt = (inst & 0x03C0) >> 6;
   op =  (inst & 0x0400) >> 10;

   switch (op)
   {
   case 0: /* LDCR */
   case 1: /* STCR */
      break;
   default:
      return (-1);
   }

   sprintf (outbuf, "\t%s\t", ops4op[op]);
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   sprintf (&outbuf[strlen(outbuf)], "%d", cnt);

   return (0);
}

/***********************************************************************
* proc5op - Process type 5 operations.
***********************************************************************/

static int
proc5op (uint16 inst)
{
   uint16 reg;
   uint16 cnt;
   uint16 op;

   reg = inst & 0x000F;
   cnt = (inst & 0x00F0) >> 4;
   op = (inst & 0x0300) >> 8;

   switch (op)
   {
   case 0: /* SRA */
   case 1: /* SRL */
   case 2: /* SLA */
   case 3: /* SRC */
      break;
   default:
      return (-1);
   }

   sprintf (outbuf, "\t%s\tR%d,%d", ops5op[op], reg, cnt);
   return (0);
}

/***********************************************************************
* proc6op - Process type 6 operations.
***********************************************************************/

static int
proc6op (uint16 inst)
{
   char *opptr;
   uint16 sr;
   uint16 st;
   uint16 op;

   sr = inst & 0x000F;
   st = (inst & 0x0030) >> 4;

   if ((inst & 0x0F00) >= 0x0C00) /* Floating point instructions */
   {
      op = (inst & 0x03C0) >> 6;
      switch (op)
      {
      case 1:  /* AR   */
      case 2:  /* CIR  */
      case 3:  /* SR   */
      case 4:  /* MR   */
      case 5:  /* DR   */
      case 6:  /* LR   */
      case 7:  /* STR  */
      case 9:  /* AD   */
      case 10: /* CID  */
      case 11: /* SD   */
      case 12: /* MD   */
      case 13: /* DD   */
      case 14: /* LD   */
      case 15: /* STD  */
	 if (model < 12)
	    return (-1);
         break;
      default:
	 return (-1);
      }
      opptr = ops6fop[op];
   }
   else if ((inst & 0x0F00) == 0x0100) 
   {
      op = (inst & 0x00C0) >> 6;
      switch (op)
      {
      case 1:  /* BIND */
      case 2:  /* DIVS */
      case 3:  /* MPYS */
	 if (model < 11)
	    return (-1);
         break;
      default:
	 return (-1);
      }
      opptr = ops6eop[op];
   }
   else
   {
      op = (inst & 0x03C0) >> 6;
      switch (op)
      {
      case 0:  /* BLWP */
      case 1:  /* B    */
      case 2:  /* X    */
      case 3:  /* CLR  */
      case 4:  /* NEG  */
      case 5:  /* INV  */
      case 6:  /* INC  */
      case 7:  /* INCT */
      case 8:  /* DEC  */
      case 9:  /* DECT */
      case 10: /* BL   */
      case 11: /* SWPB */
      case 12: /* SETO */
      case 13: /* ABS  */
         break;
      case 14: /* LDS  */
      case 15: /* LDD  */
	 if (model < 10)
	    return (-1);
	 break;
      default:
	 return (-1);
      }
      opptr = ops6op[op];
   }

   sprintf (outbuf, "\t%s\t", opptr);
   if (genoperand (sr, st, FALSE) < 0) return (-1);

   return (0);
}

/***********************************************************************
* proc7op - Process type 7 operations.
***********************************************************************/

static int
proc7op (uint16 inst)
{
   char *opptr;
   uint16 op;
   uint16 sop;

   op =  (inst & 0x00F0) >> 4;
   sop = inst & 0x000F;

   switch (op)
   {
   case 0:
      switch (sop)
      {
      case 0:  /* CRI  */
      case 1:  /* CDI  */
      case 2:  /* NEGR */
      case 3:  /* NEGD */
      case 4:  /* CRE  */
      case 5:  /* CDE  */
      case 6:  /* CER  */
      case 7:  /* CED  */
      case 14: /* XIT  */
      case 15: /* XIT  */
	 if (model < 12)
	    return (-1);
         break;
      default:
         return (-1);
      }
      opptr = ops7fop[sop];
      break;
   case 2:
      switch (sop)
      {
      case 13: /* EMD  */
      case 14: /* EINT */
      case 15: /* DINT */
	 if (model < 12)
	    return (-1);
      default:
         return (-1);
      }
      opptr = ops7eop[sop];
      break;
   case 4:  /* IDLE */
   case 6:  /* RSET */
   case 8:  /* RTWP */
   case 10: /* CKON */
   case 12: /* CKOF */
   case 14: /* LREX */
      if (sop)
         return (-1);
      opptr = ops7op[op];
      break;
   default:
      return (-1);
   }

   sprintf (outbuf, "\t%s", opptr);
   return (0);
}

/***********************************************************************
* proc8op - Process type 8 operations.
***********************************************************************/

static int
proc8op (uint16 inst)
{
   Memory *lclctl;
   SymNode *s;
   int dsegdata;
   uint16 r;
   uint16 op;

   lclctl = getctl();
   s = NULL;
   dsegdata = FALSE;

   r = inst & 0x000F;
   op = (inst & 0x00F0) >> 4;

   if ((inst & 0xFFF0) == 0x00B0) /* BLSK */
   {
      if (model < 12)
         return (-1);
      printf ("\tBLSK\t%d", r);
      return (0);
   }
   else if (inst == 0x0300) /* LIMI */
   {
      sprintf (outbuf, "\tLIMI\t%d", getinst());
      goto CHECK_LABEL;
   }

   else switch (op)
   {
   case 0:  /* LI   */
   case 2:  /* AI   */
   case 4:  /* ANDI */
   case 6:  /* ORI  */
   case 8:  /* CI   */
      sprintf (outbuf, "\t%s\tR%d,", ops8op[op], r);
      break;
   case 14: /* LWPI */
      if (r)
         return (-1);
      sprintf (outbuf, "\t%s\t", ops8op[op]);
      break;
   case 10: /* STWP */
   case 12: /* STST */
      sprintf (outbuf, "\t%s\tR%d", ops8op[op], r);
      return (0);
   default:
      return (-1);
   }

   op = getinst();
#ifdef DEBUGOPS
   fprintf (stderr, "   immediate operand, pc = >%04X, tag = %c, op = >%04X\n",
	    pcreg-2, lclctl->tag, op);
#endif
   if (lclctl->tag == EXTNDX_TAG)
      s = extlookup (op, TRUE);
   else if (!lclctl->external && lclctl->relocatable)
      s = vallookup (op, FALSE, lclctl->tag == DSEGDATA_TAG ? TRUE : FALSE,
		     csegndx);
   else if (lclctl->external)
      s = extlookup (op, FALSE);

   /* if (lclctl->relocatable)
      s = vallookup (op, TRUE, segtype, csegndx); */

   if (s)
      sprintf (&outbuf[strlen(outbuf)], "%s", s->symbol);
   else if (!dsegdata)
      sprintf (&outbuf[strlen(outbuf)], ">%04X", op);

CHECK_LABEL:
   if ((s = vallookup (pcreg-2, FALSE, segtype, csegndx)) != NULL)
      srcaddre = pcreg - 2;
   if ((s = vallookup (pcreg-1, FALSE, segtype, csegndx)) != NULL)
      srcaddro = pcreg - 1;

   return (0);
}

/***********************************************************************
* proc9op - Process type 9 operations.
***********************************************************************/

static int
proc9op (uint16 inst)
{
   uint16 st;
   uint16 sr;
   uint16 r;
   uint16 op;

   sr =  inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   r = (inst & 0x03C0) >> 6;
   op = (inst & 0x0400) >> 10;

   switch (op)
   {
   case 0: /* MPY */
   case 1: /* DIV */
      break;
   default:
      return (-1);
   }

   sprintf (outbuf, "\t%s\t", ops9op[op]);
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   sprintf (&outbuf[strlen(outbuf)], "R%d", r);

   return (0);
}

/***********************************************************************
* proc10op - Process type 10 operations.
***********************************************************************/

static int
proc10op (uint16 inst)
{
   uint16 r;
   uint16 mf;

   if (model < 10)
      return (-1);

   r =  inst & 0x000F;
   mf = (inst & 0x0010) >> 4;

   sprintf (outbuf, "\tLMF\tR%d,%d", r, mf);

   return (0);
}

/***********************************************************************
* proc11op - Process type 11 operations.
***********************************************************************/

static int
proc11op (uint16 inst)
{
   char *opptr;
   uint16 inst1;
   uint16 dt;
   uint16 dr;
   uint16 st;
   uint16 sr;
   uint16 bc;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   inst1 = getinst();
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   bc = (inst1 & 0xF000) >> 12;

   if (inst == 0xC08) /* NRM */
   {
      opptr = "NRM";
   }
   else
   {
      op = (inst & 0x003F) - 0x001E;
      switch (op)
      {
      case 0:  /* RTO  */
      case 1:  /* LTO  */
      case 2:  /* CNTO */
      case 5:  /* BDC  */
      case 6:  /* DBC  */
      case 7:  /* SWPM */
      case 8:  /* XORM */
      case 9:  /* ORM  */
      case 10: /* ANDM */
      case 11: /* SM   */
      case 12: /* AM   */
         break;
      default:
	 return (-1);
      }
      opptr = ops11op[op];
   }

   sprintf (outbuf, "\t%s\t", opptr);
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   if (genoperand (dr, dt, TRUE) < 0) return (-1);
   sprintf (chop, "%d", bc);
   strcat (outbuf, chop);

   return (0);
}

/***********************************************************************
* proc12op - Process type 12 operations.
***********************************************************************/

static int
proc12op (uint16 inst)
{
   char *opptr;
   uint16 inst1;
   uint16 ckpt;
   uint16 dt;
   uint16 dr;
   uint16 st;
   uint16 sr;
   uint16 bc;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   ckpt = inst & 0x000F;
   inst1 = getinst();
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   bc = (inst1 & 0xF000) >> 12;

   if ((inst & 0x0F00) == 0x0E00)
   {
      op = (inst & 0x00F0) >> 4;
      switch (op)
      {
      case 1:  /* SNEB */
      case 2:  /* CRC  */
      case 3:  /* TC   */
         break;
      default:
         return (-1);
      }
      opptr = ops12aop[op];
   }
   else
   {
      op = ((inst & 0x00F0) - 0x0040) >> 4;
      switch (op)
      {
      case 0:  /* CS   */
      case 1:  /* SEQB */
      case 2:  /* MOVS */
      case 8:  /* MVSR */
      case 9:  /* MVSK */
      case 10: /* POPS */
      case 11: /* PSHS */
         break;
      default:
         return (-1);
      }
      opptr = ops12bop[op];
   }

   sprintf (outbuf, "\t%s\t", opptr);
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   if (genoperand (dr, dt, TRUE) < 0) return (-1);
   sprintf (chop, "%d,R%d", bc, ckpt);
   strcat (outbuf, chop);

   return (0);
}

/***********************************************************************
* proc13op - Process type 13 operations.
***********************************************************************/

static int
proc13op (uint16 inst)
{
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 bc;
   uint16 ct;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   inst1 = getinst();
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   ct = (inst1 & 0x03C0) >> 6;
   bc = (inst1 & 0xF000) >> 12;

   op = (inst & 0x000F) - 0x000C;
   switch (op)
   {
   case 0:  /* SRAM */
   case 1:  /* SLAM */
      break;
   default:
      return (-1);
   }

   sprintf (outbuf, "\t%s\t", ops13op[op]);
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   sprintf (chop, "%d,%d", ct, bc);
   strcat (outbuf, chop);

   return (0);
}

/***********************************************************************
* proc14op - Process type 14 operations.
***********************************************************************/

static int
proc14op (uint16 inst)
{
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 ps;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   inst1 = getinst();
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   ps = (inst1 & 0xFFC0) >> 6;

   op = (inst & 0x000F) - 0x0009;
   switch (op)
   {
   case 0:  /* TMB  */
   case 1:  /* TCMB */
   case 2:  /* TSMB */
      break;
   default:
      return (-1);
   }

   sprintf (outbuf, "\t%s\t", ops14op[op]);
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   sprintf (chop, "%d", ps);
   strcat (outbuf, chop);

   return (0);
}

/***********************************************************************
* proc15op - Process type 15 operations.
***********************************************************************/

static int
proc15op (uint16 inst)
{
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 wd;
   uint16 ps;
   char chop[16];

   if (model < 12)
      return (-1);

   wd = inst & 0x000F;
   inst1 = getinst();
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   ps = (inst1 & 0xF000) >> 12;

   sprintf (outbuf, "\t%s\t", "IOF");
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   sprintf (chop, "(%d,%d)", ps, wd);
   strcat (outbuf, chop);

   return (0);
}

/***********************************************************************
* proc16op - Process type 16 operations.
***********************************************************************/

static int
proc16op (uint16 inst)
{
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 dt;
   uint16 dr;
   uint16 wd;
   uint16 ps;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   wd = inst & 0x000F;
   inst1 = getinst();
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   ps = (inst1 & 0xF000) >> 12;

   op = (inst & 0x000F) - 0x0009;
   switch (op)
   {
   case 1:  /* INSF */
   case 2:  /* XV   */
   case 3:  /* XF   */
      break;
   default:
      return (-1);
   }

   sprintf (outbuf, "\t%s\t", ops16op[op]);
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   if (genoperand (dr, dt, TRUE) < 0) return (-1);
   sprintf (chop, "(%d,%d)", ps, wd);
   strcat (outbuf, chop);

   return (0);
}

/***********************************************************************
* proc17op - Process type 17 operations.
***********************************************************************/

static int
proc17op (uint16 inst)
{
   uint16 inst1;
   uint16 disp;
   uint16 r;
   uint16 c;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   inst1 = getinst();
   disp =  inst1 & 0x00FF;
   r = (inst1 & 0x0F00) >> 8;
   c = (inst1 & 0xF000) >> 12;

   op = (inst & 0x000F) - 0x000C;
   switch (op)
   {
   case 0:  /* SRJ  */
   case 1:  /* ARJ  */
      break;
   default:
      return (-1);
   }

   sprintf (outbuf, "%-8s ", ops17op[op]);
   sprintf (chop, ">%X,%d,R%d", disp, c, r);
   strcat (outbuf, chop);

   return (0);
}

/***********************************************************************
* proc18op - Process type 18 operations.
***********************************************************************/

static int
proc18op (uint16 inst)
{
   uint16 r;
   uint16 op;

   r =  inst & 0x000F;
   op = ((inst & 0x00F0) >> 4) - 0x0003;
   switch (op)
   {
   case 0:  /* STPC */
      if (model < 12)
	 return (-1);
      break;
   case 4:  /* LIM  */
      if (model < 12)
	 return (-1);
      break;
   case 5:  /* LST  */
      if (model < 11)
	 return (-1);
      break;
   case 6:  /* LWP  */
      if (model < 11)
	 return (-1);
      break;
   case 7:  /* LCS  */
      if (model < 12)
	 return (-1);
      break;
   default:
      return (-1);
   }

   sprintf (outbuf, "\t%s\tR%d", ops18op[op], r);

   return (0);
}

/***********************************************************************
* proc19op - Process type 19 operations.
***********************************************************************/

static int
proc19op (uint16 inst)
{
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 dt;
   uint16 dr;

   if (model < 12)
      return (-1);

   inst1 = getinst();
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;

   sprintf (outbuf, "\t%s\t", "MOVA");
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   if (genoperand (dr, dt, FALSE) < 0) return (-1);
   return (0);
}

/***********************************************************************
* proc20op - Process type 20 operations.
***********************************************************************/

static int
proc20op (uint16 inst)
{
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 dt;
   uint16 dr;
   uint16 cc;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   inst1 = getinst();
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   cc = (inst1 & 0xF000) >> 12;

   op = inst & 0x000F;
   switch (op)
   {
   case 1:  /* SLSL */
   case 2:  /* SLSP */
      break;
   default:
      return (-1);
   }

   sprintf (outbuf, "\t%s\t", ops20op[op]);
   sprintf (chop, "%s,", ops20cc[cc]);
   strcat (outbuf, chop);
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   if (genoperand (dr, dt, FALSE) < 0) return (-1);

   return (0);
}

/***********************************************************************
* proc21op - Process type 21 operations.
***********************************************************************/

static int
proc21op (uint16 inst)
{
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 sl;
   uint16 dt;
   uint16 dr;
   uint16 dl;
   char chop[16];

   if (model < 12)
      return (-1);

   dl = inst & 0x000F;
   inst1 = getinst();
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   sl = (inst1 & 0xF000) >> 12;

   sprintf (outbuf, "\t%s\t", "EP");
   if (genoperand (sr, st, TRUE) < 0) return (-1);
   if (genoperand (dr, dt, TRUE) < 0) return (-1);
   sprintf (chop, "%d,%d", sl, dl);
   strcat (outbuf, chop);

   return (0);
}

/***********************************************************************
* procinst - Run an instruction.
***********************************************************************/

static void 
procinst (uint16 inst, uint16 curpc)
{
   int type;
   int error;
   uint8 ic1, ic2;

   srcaddre = dstaddre = 0;
   srcaddro = dstaddro = 0;

   error = -1;
   if (!segtype && ((inst & 0x0FFF) != 0x0FFF))
   {
      type = decodeinst (inst);

      if (type < MAX_INST_TYPES)
	 error = inst_proc[type].proc (inst);
   }

   if (error < 0)
      sprintf (outbuf, "\tDATA\t>%X", inst);

   printf ("%s", outbuf);
#if 1
   ic1 = (inst >> 8) & 0xFF;
   ic2 = inst & 0xFF;
   if (isprint (ic1) && !islower(ic1) && isprint(ic2) && !islower(ic2))
   {
      printf ("\t\tTEXT\t'");
      if (ic1 == '\'')
         printf ("''");
      else
         printf ("%c", ic1);
      if (ic2 == '\'')
         printf ("'''");
      else
         printf ("%c'", ic2);
   }
#endif
   printf ("\n");

   if (srcaddre && dstaddre)
   {
      printvals (srcaddre, -4);
      printvals (dstaddre, -2);
   }
   else if (srcaddre)
      printvals (srcaddre, -2);
   else if (dstaddre)
      printvals (dstaddre, -2);

   if (srcaddro && dstaddro)
   {
      printvals (srcaddro, -3);
      printvals (dstaddro, -1);
   }
   else if (srcaddro)
      printvals (srcaddro, -1);
   else if (dstaddro)
      printvals (dstaddro, -1);

}

/***********************************************************************
* disprog - Disassemble a program.
***********************************************************************/

void
disprog (int inseg, int segndx)
{
   SymNode *s;
   int length;
   uint16 inst;
   uint16 curpc;
   char lclidt[MAXSYMLEN+2];

#ifdef DEBUGOPS
   fprintf (stderr,
	    "disprog: pcreg = %d, pgmlen = %d, inseg = %s, segndx = %d\n",
	    pcreg, pgmlen,
	    inseg == DSEG ? "DSEG" : inseg == CSEG ? "CSEG" : "P/R",
	    segndx);
#endif

   strcpy (lclidt, idtbuf);
   if (strlen (lclidt) > 3) lclidt[3] = '\0';

   segtype = inseg;
   csegndx = segndx;

   if (segtype == DSEG)
   {
      length = dseglen;
      printf ("*\n");
      printf ("\tDSEG\n");
   }
   else if (segtype == CSEG)
   {
      length = cseglen[csegndx];
      printf ("*\n");
      if (!strcmp (csegname[csegndx], "$BLANK"))
	 printf ("\tCSEG\n");
      else
	 printf ("\tCSEG\t'%s'\n", csegname[csegndx]);
   }
   else
   {
      length = pgmlen;
      curpc = 0;
      if (length && memctl[curpc].tag == 0)
      {
	 while (memctl[curpc].tag == 0) curpc += 2;
	 printf ("\tAORG\t>%X\n", curpc);
	 pcreg = curpc;
      }
   }

   while (pcreg < length)
   {
      SymNode *sym;
      Memory *instctl;

      curpc = pcreg;

      printvals (pcreg, 0);
      printvals (pcreg+1, 1);

      instctl = getctl();
      inst = getinst();

      if (instctl->tag == ABSDATA_TAG)
      {
#ifdef DEBUGOPS
	 fprintf (stderr, "%04X  %04X   ABSDATA\n", curpc, inst);
#endif
	 procinst (inst, curpc);

      }
      else if (instctl->tag == RELDATA_TAG || instctl->tag == CMNDATA_TAG)
      {

#ifdef DEBUGOPS
	 fprintf (stderr, "%04X  %04X   RELDATA\n", curpc, inst);
#endif
	 if (instctl->external)
	    sym = extlookup (inst, FALSE);
	 else 
	    sym = vallookup (inst, TRUE, segtype, csegndx);

	 if (sym)
	    printf ("\tDATA\t%s\n", sym->symbol);
	 else
	    printf ("\tDATA\t>%04X\n", inst);
      }
      else if (instctl->tag == RELORG_TAG)
      {
	 uint16 bsssize;
	 uint16 bsspc;

#ifdef DEBUGOPS
	 fprintf (stderr, "%04X  %04X   RELORG\n", curpc, inst);
#endif
	 bsssize = inst - curpc;
	 for (bsspc = pcreg; bsspc < inst; bsspc+=2)
	 {
	    if ((sym = vallookup (bsspc, FALSE, segtype, csegndx)) != NULL)
	    {
	       printf ("%s\tEQU\t$+%d\n", sym->symbol, bsspc-curpc);
	    }
	 }
	 printf ("\tBSS\t%d\n", bsssize);
	 pcreg = inst;
      }
      else if (instctl->tag == ABSORG_TAG)
      {
	 uint16 bsssize;
	 uint16 bsspc;

#ifdef DEBUGOPS
	 fprintf (stderr, "%04X  %04X   ABSORG\n", curpc, inst);
#endif
	 bsssize = inst - curpc;
	 for (bsspc = pcreg; bsspc < inst; bsspc+=2)
	 {
	    if ((sym = vallookup (bsspc, FALSE, segtype, csegndx)) != NULL)
	    {
	       printf ("%s\tEQU\t$+%d\n", sym->symbol, bsspc-curpc);
	    }
	 }
	 printf ("\tAORG\t>%X\n", inst);
	 pcreg = inst;
      }
      else if (instctl->tag == EXTNDX_TAG)
      {
	 sym = extlookup (inst, TRUE);

#ifdef DEBUGOPS
	 fprintf (stderr, "%04X  %04X   EXTNDX\n", curpc, inst);
#endif
	 if (sym)
	 {
	    printf ("\tDATA\t%s", sym->symbol);
	    if (instctl->value) printf ("+%d", instctl->value);
	    printf ("\n");
	 }
	 else
	    printf ("\tDATA\t>%04X\n", inst);
      }
      else if (instctl->tag == DSEGDATA_TAG)
      {
#ifdef DEBUGOPS
	 fprintf (stderr, "%04X  %04X   DSEGDATA\n", curpc, inst);
#endif
	 s = vallookup (inst, FALSE, segtype, csegndx);
	 if (s)
	    printf ("\tDATA\t%s\n", s->symbol);
	 else
	    printf ("\tDATA\t>%04X\n", inst);
      }
   }
   printvals (pcreg, 0);

   if (segtype == DSEG)
   {
      printf ("\tDEND\n");
      printf ("*\n");
   }
}
