/***********************************************************************
*
* DUMPDISK - Dumps a TI990 tiline disk image file.
*
* Changes:
*      07/20/04   DGP   Adapted from dumpfpy.
*      10/07/18   DGP   Fixed overhead.
*
***********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stddef.h>
#include <errno.h>
#include <ctype.h>

#define NORMAL		0
#define ABORT		16

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#include "simdsk.h"

#include "hexdump.h"
#include "support.h"

/***********************************************************************
* main
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *fd;
   char *fname;
   char *model;
   long dskloc;
   int i;
   int j;
   int cyls, heads, sectrk, overhead, sectlen;
   int tracklen;
   int realdev;
   int head, cyl;
   int start, end;
   char buf[512];
   
   fname = NULL;
   model = NULL;
   cyl = 0;
   head = 0;
   start = 0;
   end = 0;
   j = 0;
   realdev = 0;
   for (i = 1; i < argc; i++)
   {
      if (argv[i][0] == '-')
      {
         if (argv[i][1] == 'm')
	 {
	    i++;
	    model = argv[i];
	 }
	 else
	    goto USAGE;
      }
      else switch (j++)
      {
      case 0:
         fname = argv[i];
	 break;
      case 1:
         cyl = atoi(argv[i]);
	 break;
      case 2:
         head = atoi(argv[i]);
	 break;
      case 3:
         start = atoi(argv[i]);
	 end = start;
	 break;
      case 4:
         end = atoi(argv[i]);
	 break;
      default:
      USAGE:
	 fprintf (stderr,
      "usage: dumpdisk [-m model] disk.file [cyl head [startsec [endsec]]]\n");
	 exit (ABORT);
      }
   }
   if (fname == NULL) goto USAGE;

   if (!strncmp (fname, "/dev/", 5))
   {
      if (model == NULL)
      {
	 fprintf (stderr, "You must specify a model for a real device.\n");
	 goto USAGE;
      }
      realdev = 1;
   }

   /*
   ** Open the disk image file
   */

   if ((fd = fopen (fname, "rb")) == NULL)
   {
      perror ("Can't open disk file");
      exit (ABORT);
   }

   /*
   ** Setup/Read the disk geometry
   */

   if (!realdev)
   {
      if ((cyls = dskreadint (fd)) < 0)
      {
	 perror ("Can't read disk geometry");
	 exit (ABORT);
      }
      if ((heads = dskreadint (fd)) < 0) 
      {
	 perror ("Can't read disk geometry");
	 exit (ABORT);
      }
      heads &= 0xFFFF;
      if ((sectrk = dskreadint (fd)) < 0)
      {
	 perror ("Can't read disk geometry");
	 exit (ABORT);
      }
      overhead = (sectrk >> 8) & 0x3FF;
      sectrk = sectrk & 0xFF;
      if ((sectlen = dskreadint (fd)) < 0)
      {
	 perror ("Can't read disk geometry");
	 exit (ABORT);
      }

      tracklen = sectrk * (sectlen + overhead);

      for (i = 0; i < MAXDISKS; i++)
      {
	 if (disks[i].cyls == cyls &&
	     disks[i].heads == heads &&
	     disks[i].sectrk == sectrk &&
	     disks[i].bytsec == sectlen &&
	     disks[i].overhead == overhead)
	    break;
      }
      if (i == MAXDISKS)
      {
	 fprintf (stderr, "Unknown Disk model*\n");
      }
   }
   else
   {
      for (i = 0; i < MAXDISKS; i++)
      {
	 if (!strcmp (disks[i].model, model))
	    break;
      }
      if (i == MAXDISKS)
      {
	 fprintf (stderr, "Unknown Disk model*\n");
	 exit (ABORT);
      }
      cyls = disks[i].cyls;
      heads = disks[i].heads;
      sectrk  = disks[i].sectrk;
      sectlen = disks[i].bytsec;
      overhead = disks[i].overhead;
      tracklen = sectrk * (sectlen + overhead);
   }
	       
   printf ("%s disk geometry:\n", disks[i].model);
   printf ("   cyls      = %d\n", cyls);
   printf ("   heads     = %d\n", heads);
   printf ("   sectrk    = %d\n", sectrk);
   printf ("   sectlen   = %d\n", sectlen);
   printf ("   overhead  = %d\n", overhead);
   printf ("   tracklen  = %d\n", tracklen);
   printf ("   disk size = %d bytes\n", tracklen * heads * cyls);

   if (cyl >= cyls)
   {
      printf ("Invalid disk cylinder specified, abort\n");
      exit (ABORT);
   }
   if (head >= heads)
   {
      printf ("Invalid disk head specified, abort\n");
      exit (ABORT);
   }
   if (start >= sectrk)
   {
      printf ("Invalid disk start sector specified, abort\n");
      exit (ABORT);
   }
   if (end >= sectrk)
   {
      printf ("Invalid disk end sector specified, abort\n");
      exit (ABORT);
   }

   for (j = start; j <= end; j++)
   {
      /*
      ** Calculate location
      */

      dskloc = (tracklen * head) + (tracklen * heads * cyl) +
	 ((overhead + sectlen) * j) + (realdev ? 0 : DSKOVERHEAD);

#ifdef DEBUG
      printf ("   cyl = %d, head = %d, sector = %d\n",
	       cyl, head, start);
      printf ("   dskloc = %ld\n", dskloc);
#endif

      /*
      ** Go to the specified location on disk
      */

      if (fseek (fd, dskloc, SEEK_SET) < 0)
      {
	 perror ("seek failed");
	 exit (ABORT);
      }

      /*
      ** Read sector
      */

      for (i = 0; i < sectlen; i++)
	 buf[i] = fgetc (fd);
      /*
      ** Dump it
      */
      printf ("\ncyl %d head %d sect %d dskloc = %ld:\n", cyl, head, j, dskloc);
      HEXDUMP (stdout, buf, sectlen, 0);
   }

   return (NORMAL);
}
