/***********************************************************************
*
* getdisk - Gets the disk image from the TI computer. Use in
*           conjunction with the 990 program ddisk.asm
*
* Changes:
*      06/30/04   DGP   Original
*      02/17/12   DGP   Changed to read/write a track at a time.
*
***********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <termios.h>

#define NORMAL	0
#define ABORT	16

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#include "simdsk.h"
#include "support.h"

struct termios cmdtty, runtty;
int fd;

/*
** Terminal routines borrowed from Bub Supnik's simh
*/

static int
ttinit (FILE *ifd)
{
   fd = fileno(ifd);
   if (tcgetattr (fd, &cmdtty) < 0) return -1;		/* get old flags */
   runtty = cmdtty;
   runtty.c_lflag = runtty.c_lflag & ~(ECHO | ICANON);	/* no echo or edit */
   runtty.c_oflag = runtty.c_oflag & ~OPOST;		/* no output edit */
   runtty.c_iflag = runtty.c_iflag & ~ICRNL;		/* no cr conversion */
   runtty.c_cc[VINTR] = 0;				/* interrupt */
   runtty.c_cc[VQUIT] = 0;				/* no quit */
   runtty.c_cc[VERASE] = 0;
   runtty.c_cc[VKILL] = 0;
   runtty.c_cc[VEOF] = 0;
   runtty.c_cc[VEOL] = 0;
   runtty.c_cc[VSTART] = 0;				/* no host sync */
   runtty.c_cc[VSUSP] = 0;
   runtty.c_cc[VSTOP] = 0;
#if defined (VREPRINT)
   runtty.c_cc[VREPRINT] = 0;				/* no specials */
#endif
#if defined (VDISCARD)
   runtty.c_cc[VDISCARD] = 0;
#endif
#if defined (VWERASE)
   runtty.c_cc[VWERASE] = 0;
#endif
#if defined (VLNEXT)
   runtty.c_cc[VLNEXT] = 0;
#endif
   runtty.c_cc[VMIN] = 0;				/* no waiting */
   runtty.c_cc[VTIME] = 0;
#if defined (VDSUSP)
   runtty.c_cc[VDSUSP] = 0;
#endif
#if defined (VSTATUS)
   runtty.c_cc[VSTATUS] = 0;
#endif
   return 0;
}

static int
ttrunstate (void)
{
   runtty.c_cc[VINTR] = 0;
   if (tcsetattr (fd, TCSAFLUSH, &runtty) < 0) return -1;
   return 0;
}

static int
ttcmdstate (void)
{
   if (tcsetattr (fd, TCSAFLUSH, &cmdtty) < 0) return -1;
   return 0;
}

static int
ttclose (void)
{
   return ttcmdstate ();
}

static int
poll_kbd (void)
{
   int status;
   unsigned char buf[1];

   status = read (fd, buf, 1);
   if (status != 1) return -1;
   else return (buf[0]);
}

static int 
readint (void)
{
   int a, b;
   while ((a = poll_kbd()) == -1) ;
   while ((b = poll_kbd()) == -1) ;
   return (a << 8 | b);
}

/***********************************************************************
* Main program.
***********************************************************************/

int
main(int argc, char **argv)
{
   int  *memory;
   FILE *iterm;
   FILE *oterm;
   FILE *ofd;
   off_t curloc, newloc;
   int oldcyl = -1;
   int diskndx;
   int disksize;
   int i;
   int tracksize;
   int status;
   int dskinfo[6];

   /*
   ** Check args
   */

   if (argc != 3)
   {
      fprintf (stderr, "usage: getdisk model disk.file\n");
      exit (ABORT);
   }

   /*
   ** Check disk model
   */

   for (diskndx = 0; diskndx < MAXDISKS; diskndx++)
   {
      if (!strcmp (disks[diskndx].model, argv[1])) break;
   }
   if (diskndx == MAXDISKS)
   {
      fprintf (stderr, "getdisk: Unknown disk model: %s\n", argv[1]);
      exit (ABORT);
   }

   /*
   ** Calculate the size of the disk and track for display
   */

   tracksize = (disks[diskndx].overhead + disks[diskndx].bytsec) *
	       disks[diskndx].sectrk;
   disksize = tracksize * disks[diskndx].heads * disks[diskndx].cyls;

   printf ("%s: disksize = %d, tracksize = %d\n\n",
	    argv[1], disksize, tracksize);

   /*
   ** Allocate the needed memory
   */

   if ((memory = (int *)malloc (tracksize * sizeof(int))) == NULL)
   {
      fprintf (stderr, "getdisk: Can't malloc memory\n");
      exit (ABORT);
   }

   /*
   ** Open the disk image file
   */

   if ((ofd = fopen (argv[2], "wb")) == NULL)
   {
      perror ("Can't open file");
      exit (ABORT);
   }

   /*
   ** Write the disk geometry information
   */

   dskwriteint (ofd, disks[diskndx].cyls);
   dskwriteint (ofd, disks[diskndx].heads);
   dskwriteint (ofd, (disks[diskndx].overhead << 8) | disks[diskndx].sectrk);
   dskwriteint (ofd, disks[diskndx].bytsec);

   /*
   ** Set up the port
   */

   if ((iterm = fopen ("/dev/ttyS0", "rb")) == NULL)
   {
      perror ("Can't open terminal");
      fclose (ofd);
      exit (ABORT);
   }
   if ((oterm = fopen ("/dev/ttyS0", "w")) == NULL)
   {
      perror ("Can't open terminal");
      fclose (ofd);
      exit (ABORT);
   }

   ttinit (iterm);
   ttrunstate();

   /*
   ** Tell the ddisk program to go
   */

   fputc ('\n', oterm);

   /*
   ** Read the geometry.
   */

   for (i = 0; i < 6; i++)
   {
      dskinfo[i] = readint();
   }

   /*
   ** Process the disk.
   */

   status = NORMAL;
   curloc = newloc = 0;
   while (status == NORMAL)
   {
      int c;
      int cyl, head, sect;

      /*
      ** Read the disk location and sector.
      */
      fputc ('\n', oterm);
      cyl = readint();
      head = readint();
      sect = readint();

      for (i = 0; i < disks[diskndx].bytsec; i++)
      {
	 while ((c = poll_kbd()) == -1) ;
	 memory[i] = c;
      }
      if (cyl == 0xFFFF)
         break;

      /*
      ** Calculate new location
      */

      curloc = ftell (ofd);
      newloc = (tracksize * head) +
	 (tracksize * disks[diskndx].heads * cyl) +
	 ((disks[diskndx].overhead + disks[diskndx].bytsec) * sect) +
	 DSKOVERHEAD;

#ifdef DEBUG
      printf ("curloc = %d, newloc = %d\n", curloc, newloc);
#endif
      while (curloc < newloc)
      {
	 if (fputc (0, ofd) < 0)
	 {
	    perror ("Write error");
	    status = ABORT;
	    goto DIE;
	 }
	 curloc++;
      }

      if (cyl != oldcyl)
      {
	 printf ("cyl %d\n", cyl);
	 oldcyl = cyl;
      }

      /*
      ** Dump the disk sector.
      */

      for (i = 0; i < disks[diskndx].bytsec; i++)
      {
	 if (fputc (memory[i], ofd) < 0)
	 {
	    perror ("Write error");
	    status = ABORT;
	    goto DIE;
	 }
      }

      /*
      ** If overhead is specified, pad out that amount
      */

      if (disks[diskndx].overhead)
      {
	 for (i = 0; i < disks[diskndx].overhead; i++)
	    if (fputc (0, ofd) < 0)
	    {
	       perror ("Write error");
	       status = ABORT;
	       goto DIE;
	    }
      }

   }

   fputc ('\n', oterm);
   curloc = ftell (ofd);
   newloc = (tracksize * disks[diskndx].heads) +
      (tracksize * disks[diskndx].heads * disks[diskndx].cyls) +
      ((disks[diskndx].overhead + disks[diskndx].bytsec) *
      disks[diskndx].sectrk) + DSKOVERHEAD;

#ifdef DEBUG
   printf ("curloc = %lld, newloc = %lld\n", curloc, newloc);
#endif
   while (curloc < newloc)
   {
      if (fputc (0, ofd) < 0)
      {
	 perror ("Write error");
	 status = ABORT;
	 goto DIE;
      }
      curloc++;
   }

   /*
   ** Clean up and quit
   */

DIE:
   ttcmdstate();
   ttclose();

   fclose (iterm);
   fclose (oterm);
   fclose (ofd);
   return (status);
}
