/***********************************************************************
*
* HD - Hexdump a disk/floppy/tape image.
*
* Changes:
*   03/31/05   DGP   Original
*   01/03/11   DGP   Added tape mode dump.
*   10/07/18   DGP   Fixed overhead.
*   12/09/22   DGP   Added EBCDIC mode.
*
***********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/param.h>

#define MAXMEMSIZE 1024
#define MAXTAPERECLEN 32768

#include "hexdump.h"
#include "support.h"
#include "simdsk.h"

/*
** EBCDIC to ASCII conversion table.
*/

static unsigned char ebcasc[256] =
{
 /*00  NU    SH    SX    EX    PF    HT    LC    DL */
      0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*08              SM    VT    FF    CR    SO    SI */
      0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*10  DE    D1    D2    TM    RS    NL    BS    IL */
      0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*18  CN    EM    CC    C1    FS    GS    RS    US */
      0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*20  DS    SS    FS          BP    LF    EB    EC */
      0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*28              SM    C2    EQ    AK    BL       */
      0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*30              SY          PN    RS    UC    ET */
      0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*38                    C3    D4    NK          SU */
      0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*40  SP                                           */
      0x20, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*48             CENT    .     <     (     +     | */
      0x2e, 0x2e, 0x9B, 0x2E, 0x3C, 0x28, 0x2B, 0x7C,
 /*50   &                                           */
      0x26, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*58               !     $     *     )     ;     ^ */
      0x2e, 0x2e, 0x21, 0x24, 0x2A, 0x29, 0x3B, 0x5E,
 /*60   -     /                                     */
      0x2D, 0x2F, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*68               |     ,     %     _     >     ? */
      0x2e, 0x2e, 0x7C, 0x2C, 0x25, 0x5F, 0x3E, 0x3F,
 /*70                                               */
      0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*78         `     :     #     @     '     =     " */
      0x2e, 0x60, 0x3A, 0x23, 0x40, 0x27, 0x3D, 0x22,
 /*80         a     b     c     d     e     f     g */
      0x2e, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
 /*88   h     i           {                         */
      0x68, 0x69, 0x2e, 0x7B, 0x2e, 0x2e, 0x2e, 0x2e,
 /*90         j     k     l     m     n     o     p */
      0x2e, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70,
 /*98   q     r           }                         */
      0x71, 0x72, 0x2e, 0x7D, 0x2e, 0x2e, 0x2e, 0x2e,
 /*A0         ~     s     t     u     v     w     x */
      0x2e, 0x7E, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
 /*A8   y     z                       [             */
      0x79, 0x7A, 0x2e, 0x2e, 0x2e, 0x5B, 0x2e, 0x2e,
 /*B0                                               */
      0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*B8                                 ]             */
      0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x5D, 0x2e, 0x2e,
 /*C0   {     A     B     C     D     E     F     G */
      0x7B, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
 /*C8   H     I                                     */
      0x48, 0x49, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*D0   }     J     K     L     M     N     O     P */
      0x7D, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50,
 /*D8   Q     R                                     */
      0x51, 0x52, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*E0   \           S     T     U     V     W     X */
      0x5C, 0x2e, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
 /*E8   Y     Z                                     */
      0x59, 0x5A, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e,
 /*F0   0     1     2     3     4     5     6     7 */
      0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
 /*F8   8     9                                     */
      0x38, 0x39, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e, 0x2e
};

/***********************************************************************
* ebcdump - Dump EBCDIC formated data in hex.
***********************************************************************/

static void
ebcdump (FILE *file, char *ptr, int size, int offset)
{
   int jjj;
   int iii;
   unsigned char *tp;
   char *cp;

   for (iii = 0, tp = (char *)ptr, cp = (char *)ptr; iii < size; )
   {
      fprintf (file, "%04X   ", iii + offset);
      for (jjj = 0; jjj < 8; jjj++)
      {
	 if (cp < ((char *)ptr + size))
	 {
	    fprintf (file, "%02X", *cp++ & 0xFF);
	    if (cp < ((char *)ptr + size))
	    {
	       fprintf (file, "%02X ", *cp++ & 0xFF);
	    }
	    else
	    {
	       fprintf (file, "   ");
	    }
	 }
	 else
	 {
	    fprintf (file, "     ");
	 }
	 iii += 2;
      }

      fprintf (file, "   ");
      for (jjj = 0; jjj < 8; jjj++)
      {
	 if (tp < ((unsigned char *)ptr + size))
	 {
	    fprintf (file, "%c", ebcasc[*tp]);
	    tp++;
	    if (tp < ((unsigned char *)ptr + size))
	    {
	       fprintf (file, "%c", ebcasc[*tp]);
	       tp++;
	    }
	    else
	    {
	       fprintf (file, "  ");
	    }
	 }
	 else
	 {
	    fprintf (file, "   ");
	 }
      }
      fprintf (file, "\n");
   }
}

/***********************************************************************
* main
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *fd;
   char *infile;
   char *bp;
   char *mem;
   off_t seekoff;
   long seeklen;
   int i;
   int size;
   int ch;
   int usegeo;
   int reclen;
   int reccnt;
   int filecnt;
   int totcnt;
   int alloclen;
   int blkmode;
   int tapemode;
   int ebcdic;

   /*
   ** Scan off args 
   */

   infile = NULL;
   reclen = MAXMEMSIZE;
   alloclen = MAXMEMSIZE;
   seeklen = 0;
   seekoff = 0;
   blkmode = 0;
   tapemode = 0;
   usegeo = 0;
   ebcdic = 0;
   totcnt = -1;

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
         for (bp++; *bp; bp++) switch (*bp)
         {
	 case 'b':
            reclen = DEV_BSIZE;
	    blkmode = 1;
	    break;

	 case 'c':
	    i++;
	    totcnt = atoi (argv[i]);
	    break;

	 case 'e':
	    ebcdic = 1;
	    break;

	 case 'g':
	    usegeo = 1;
	    blkmode = 1;
	    break;

	 case 'r':
	    i++;
	    reclen = atoi (argv[i]);
	    break;

	 case 's':
	    i++;
	    seeklen = atol (argv[i]);
	    break;

	 case 't':
            reclen = MAXTAPERECLEN;
	    tapemode = 1;
	    break;

         default:
      USAGE:
	    printf ("usage: hd [-options] in.file \n");
            printf (" options:\n");
            printf ("    -b           - Block mode\n");
            printf ("    -c NN        - Record/Block dump count\n");
            printf ("    -e           - EBCDIC mode\n");
            printf ("    -r NN        - Record/Block length\n");
            printf ("    -g           - Use disk geometry data\n");
            printf ("    -s NN        - Seek location\n");
            printf ("    -t           - Tape dump mode\n");
	    return (1);
         }
      }
      else if (!infile)
      {
        infile = argv[i];
      }
      else
      {
         goto USAGE;
      }

   }
   if (!infile) goto USAGE;

   if ((fd = fopen (infile, "rb")) == NULL)
   {
      perror ("hd: Can't open input file");
      exit (1);
   }

   /*
   ** Read the disk geometry
   */

   if (usegeo)
   {
      int tracklen, cyls, heads, dskloc, sectrk, overhead, sectlen;

      if ((cyls = dskreadint (fd)) < 0)
      {
	 perror ("Can't read disk geometry");
	 exit (1);
      }
      if ((heads = dskreadint (fd)) < 0) 
      {
	 perror ("Can't read disk geometry");
	 exit (1);
      }
      heads &= 0xFFFF;
      if ((dskloc = dskreadint (fd)) < 0)
      {
	 perror ("Can't read disk geometry");
	 exit (1);
      }
      sectrk = dskloc & 0xFF;
      overhead = (dskloc >> 8) & 0x3FF;
      if ((sectlen = dskreadint (fd)) < 0)
      {
	 perror ("Can't read disk geometry");
	 exit (1);
      }

      tracklen = sectrk * (sectlen + overhead);

      for (i = 0; i < MAXDISKS; i++)
      {
	 if (disks[i].cyls == cyls &&
	     disks[i].heads == heads &&
	     disks[i].sectrk == sectrk &&
	     disks[i].bytsec == sectlen &&
	     disks[i].overhead == overhead)
	    break;
      }
      if (i == MAXDISKS)
      {
	 printf ("*Unknown Disk model*\n");
      }
		  
      printf ("%s disk geometry:\n", disks[i].model);
      printf ("   cyls      = %d\n", cyls);
      printf ("   heads     = %d\n", heads);
      printf ("   sectrk    = %d\n", sectrk);
      printf ("   sectlen   = %d\n", sectlen);
      printf ("   overhead  = %d\n", overhead);
      printf ("   tracklen  = %d\n", tracklen);
      printf ("   disk size = %d bytes\n", tracklen * heads * cyls);

      reclen = sectlen;
   }

   if (reclen > alloclen) alloclen = reclen;
   if ((mem = malloc (alloclen)) == NULL)
   {
      fprintf (stderr, "hd: Can't allocate memory\n");
      exit (1);
   }

   size = 0;
   reccnt = 0;
   filecnt = 0;

   printf ("Dump of %s: %s %s mode\n", infile,
	    tapemode ? "tape" : "disk",
	    blkmode ? "block" : "record");

   if (tapemode)
   {
      while (1)
      {
	 size = tapereadint (fd);
	 if (size == 0)
	 {
	    printf ("File %d Record %d: EOF\n", filecnt+1, ++reccnt);
	    reccnt = 0;
	    filecnt++;
	    continue;
	 }
	 if (size < 0) break;
	 fread (mem, size, 1, fd);
	 if (++reccnt >= seeklen)
	 {
	    printf ("File %d Record %d: Length %d\n",
		    filecnt+1, reccnt, size);
	    HEXDUMP (stdout, mem, size, 0);
	 }
	 tapereadint (fd);
	 if (totcnt >= 0 && reccnt >= totcnt)
	 {
	    break;
	 }
      }
   }
   else
   {
      if (blkmode)
      {
	 seekoff = (off_t)seeklen * (off_t)reclen;
	 if (usegeo) seekoff += DSKOVERHEAD;

	 if (fseek (fd, seekoff, SEEK_SET) < 0)
	 {
	    perror ("hd: Can't seek in file");
	    fclose (fd);
	    free (mem);
	    exit (1);
	 }
      }

      while ((ch = fgetc (fd)) != EOF)
      {
	 mem[size++] = ch % 0xFF;
	 if (size == reclen)
	 {
	    printf ("%s %ld:\n", blkmode ? "Block" : "Record",
	            reccnt + seeklen);
	    if (ebcdic)
	       ebcdump (stdout, mem, size, 0);
	    else
	       HEXDUMP (stdout, mem, size, 0);
	    size = 0;
	    reccnt++;
	    if (totcnt >= 0 && reccnt >= totcnt)
	    {
	       break;
	    }
	 }
      }

      if (size)
      {
	 printf ("%s %ld:\n", blkmode ? "Block" : "Record", reccnt + seeklen);
	 if (ebcdic)
	    ebcdump (stdout, mem, size, 0);
	 else
	    HEXDUMP (stdout, mem, size, 0);
      }
   }

   fclose (fd);
   free (mem);
   
   return (0);
}
