/***********************************************************************
*
* MKTAPE - Make a TI990 TILINE tape image file.
*
* Changes:
*   03/31/05   DGP   Original
*   02/13/14   DGP   Added -i and -o options.
*   06/01/16   DGP   Add -d option.
*   10/07/18   DGP   Fixed overhead.
*
***********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>

#include "support.h"
#include "simdsk.h"

#define NORMAL		0
#define ABORT		16

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#define BLOCKSIZE 512
#define SECSBLOCK 40

typedef struct {
   int tracklen;
   int cyls;
   int seccyl;
   int heads;
   int sectrk;
   int overhead;
   int sectlen;
} Geometry_t;

static int verbose;
static int zerolist;

/***********************************************************************
* Put short int in memory.
***********************************************************************/

static void
putshort (char *mem, int val)
{
   *mem++ = (val >> 8) & 0xFF;
   *mem = val & 0xFF;
}

/***********************************************************************
* Read the input disk geometry
***********************************************************************/

static int
processgeometry (FILE *ifd, FILE *ofd, char *mem, Geometry_t *geometry)
{
   int i;

   if ((geometry->cyls = dskreadint (ifd)) < 0)
      goto DISKERROR;
   if ((geometry->heads = dskreadint (ifd)) < 0) 
      goto DISKERROR;
   geometry->heads &= 0xFFFF;
   if ((i = dskreadint (ifd)) < 0)
      goto DISKERROR;
   geometry->sectrk = i & 0xFF;
   geometry->overhead = (i >> 8) & 0x3FF;
   if ((geometry->sectlen = dskreadint (ifd)) < 0)
      goto DISKERROR;

   geometry->tracklen = geometry->sectrk *
	    (geometry->sectlen + geometry->overhead);
   geometry->seccyl = geometry->heads * geometry->sectrk;

   for (i = 0; i < MAXDISKS; i++)
   {
      if (disks[i].cyls == geometry->cyls &&
	  disks[i].heads == geometry->heads &&
	  disks[i].sectrk == geometry->sectrk &&
	  disks[i].bytsec == geometry->sectlen &&
	  disks[i].overhead == geometry->overhead)
	 break;
   }
   if (i == MAXDISKS)
   {
      fprintf (stderr, "mktape: Unknown Disk model\n");
      return (-1);
   }
	       
   if (verbose)
   {
      printf ("%s disk geometry:\n", disks[i].model);
      printf ("   cyls      = %d\n", geometry->cyls);
      printf ("   heads     = %d\n", geometry->heads);
      printf ("   sectrk    = %d\n", geometry->sectrk);
      printf ("   sectlen   = %d\n", geometry->sectlen);
      printf ("   overhead  = %d\n", geometry->overhead);
      printf ("   tracklen  = %d\n", geometry->tracklen);
      printf ("   disk size = %d bytes, %d sectors\n",
	       geometry->tracklen * geometry->heads * geometry->cyls,
	       geometry->cyls * geometry->seccyl);
   }

   memset (mem, 0, BLOCKSIZE);
   putshort (&mem[0], geometry->cyls);
   putshort (&mem[2], geometry->heads);
   putshort (&mem[4], geometry->sectrk);
   putshort (&mem[6], geometry->sectlen);
   tapewriteint (ofd, BLOCKSIZE);
   fwrite (mem, 1, BLOCKSIZE, ofd);
   tapewriteint (ofd, BLOCKSIZE);

   return (0);

DISKERROR:
   fprintf (stderr, "mktape: Can't read disk geometry\n");
   return (-1);
}

/***********************************************************************
* process disk image to tape.
***********************************************************************/

static int
processdisk (FILE *ifd, FILE *ofd, char *mem, Geometry_t *geometry)
{
   char *bp;
   int status = 0;
   int i, j, k;
   int tapelen;
   int firstzero = FALSE;
   int totalzero = 0;
   int lastdata = 0;
   int secsrec;

   bp = mem;
   secsrec = 0;
   for (i = 0; i < geometry->cyls; i++)
   {
      for (j = 0; j < geometry->heads; j++)
      {
	 for (k = 0; k < geometry->sectrk; k++)
	 {
	    int l, zeros;

	    putshort (bp, i);
	    putshort (bp+2, j);
	    putshort (bp+4, k);
	    putshort (bp+6, geometry->sectlen);

	    if (fread (bp+8, 1, geometry->sectlen, ifd) != geometry->sectlen)
	    {
	       perror ("mktape: Can't read input disk");
	       status = -1;
	       goto DIE;
	    }
	    zeros = TRUE;
	    for (l = 0; l < geometry->sectlen; l++)
	    {
	       if (bp[l+8])
	       {
	          zeros = FALSE;
		  break;
	       }
	    }

	    if (!zeros)
	    {
	       if (firstzero)
	       {
	          firstzero = FALSE;
		  if (zerolist)
		   printf ("        end:   cyl = %4d, head = %2d, sect = %3d\n",
			   i, j, k);
	       }
	       status++;
	       bp += BLOCKSIZE;
	       if (++secsrec >= SECSBLOCK)
	       {
		  tapelen = bp - mem;
		  tapewriteint (ofd, tapelen);
		  if (fwrite (mem, 1, tapelen, ofd) != tapelen)
		  {
		     perror ("mktape: Can't write output disk");
		     status = -1;
		     goto DIE;
		  }
		  tapewriteint (ofd, tapelen);
		  bp = mem;
		  secsrec = 0;
	       }
	    }
	    else
	    {
	       if (!firstzero)
	       {
	          firstzero = TRUE;
		  lastdata = i;
		  if (zerolist)
		   printf ("   zero start: cyl = %4d, head = %2d, sect = %3d\n",
			   i, j, k);
	       }
	       totalzero++;
	    }
	 }
      }
   }
   if (secsrec)
   {
      tapelen = bp - mem;
      tapewriteint (ofd, tapelen);
      if (fwrite (mem, 1, tapelen, ofd) != tapelen)
      {
	 perror ("mktape: Can't write output disk");
	 status = -1;
	 goto DIE;
      }
      tapewriteint (ofd, tapelen);
   }
   if (firstzero && zerolist)
   {
      printf ("        end:   cyl = %4d, head = %2d, sect = %3d\n", i, j, k);
   }
   if (verbose)
   {
      printf ("   Processed %d zero sectors, lastdata = %d\n",
	    totalzero, lastdata + 1);
   }


DIE:
   return (status);
}

/***********************************************************************
* main
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *ifd;
   FILE *tfd;
   char *bp;
   char *tapefile;
   char *infile;
   int i;
   int arg;
   int done;
   int diskmode;
   int inreclen, outreclen;
   int binary;
   int noeof;
   int reccnt;
   int filecnt;
   char inbuf[32768];

   verbose = FALSE;
   zerolist = FALSE;
   diskmode = FALSE;
   binary = FALSE;
   noeof = FALSE;
   tapefile = NULL;
   infile = NULL;
   inreclen = 80;
   outreclen = 80;

   /*
   ** Scan off args 
   */

   for (arg = 1; arg < argc; arg++)
   {
      bp = argv[arg];

      if (*bp == '-')
      {
         for (bp++; *bp; bp++) switch (*bp)
         {
	 case 'd':
	    diskmode = TRUE;
	    /* fall through */

	 case 'b':
	    binary = TRUE;
	    break;

	 case 'i':
	    arg++;
	    inreclen = atoi (argv[arg]);
	    break;

	 case 'n':
	    noeof = TRUE;
	    break;

	 case 'o':
	    arg++;
	    outreclen = atoi (argv[arg]);
	    break;

	 case 'r':
	    arg++;
	    inreclen = atoi (argv[arg]);
	    outreclen = inreclen;
	    break;

	 case 'v':
	    verbose = TRUE;
	    break;

	 case 'z':
	    zerolist = TRUE;
	    break;

         default:
      USAGE:
	    printf ("usage: mktape [-options] tapefile infile...\n");
            printf (" options:\n");
	    printf ("    -b           - Binary mode\n");
	    printf ("    -d           - Disk file mode\n");
            printf ("    -i NN        - Input record length\n");
	    printf ("    -n           - No EOF between file\n");
            printf ("    -o NN        - Output record length\n");
            printf ("    -r NN        - Record length, default = 80\n");
	    return (ABORT);
         }
      }
      else if (!tapefile)
      {
        tapefile = argv[arg];
	arg++;
	break;
      }
   }

   if (!tapefile) goto USAGE;

   /*
   ** Open the tape image file
   */

   if (verbose) printf ("mktape: Opening %s for output\n", tapefile);
   if ((tfd = fopen (tapefile, "wb")) == NULL)
   {
      fprintf (stderr, "mktape: Can't open tapefile: %s: %s\n",
	       tapefile, strerror(errno));
      exit (ABORT);
   }

   filecnt = 0;
   for (; arg < argc; arg++)
   {
      /*
      ** Open the input file
      */

      infile = argv[arg];
      if (verbose) printf ("Opening %s for input\n", infile);
      if (binary)
	 ifd = fopen (infile, "rb");
      else
	 ifd = fopen (infile, "r");
      if (ifd == NULL)
      {
	 fprintf (stderr, "mktape: Can't open input: %s: %s\n",
		  infile, strerror(errno));
	 exit (ABORT);
      }

      filecnt++;
      reccnt = 0;
      if (diskmode)
      {
	 Geometry_t geometry;

	 if (processgeometry (ifd, tfd, inbuf, &geometry) == 0)
	 {
	    reccnt += processdisk (ifd, tfd, inbuf, &geometry);
	 }
      }

      else
      {
	 done = FALSE;
	 while (!done)
	 {
	    if (binary)
	    {
	       size_t inlen;

	       inlen = fread (inbuf, 1, inreclen, ifd);
	       if (inlen != inreclen)
	       {
		  done = TRUE;
	       }
	       else
	       {
		  for (i = inreclen; i < outreclen; i++)
		  {
		     inbuf[i] = 0;
		  }
	       }
	    }
	    else
	    {
	       if (fgets (inbuf, sizeof(inbuf), ifd) == NULL)
	       {
		  done = TRUE;
	       }
	       else
	       {
		  i = strlen(inbuf) - 1;
		  if (i >= inreclen)
		  {
		     inbuf[inreclen] = '\0';
		     i = inreclen;
		  }
		  for (; i < outreclen; i++)
		  {
		     inbuf[i] = ' ';
		  }
	       }
	    }

	    if (!done)
	    {
	       tapewriteint (tfd, outreclen);
	       fwrite (inbuf, outreclen, 1, tfd);
	       tapewriteint (tfd, outreclen);
	       reccnt++;
	    }
	 }
      }

      if (verbose)
         printf ("   Processed %d %s\n",
		 reccnt, diskmode ? "sectors" : "records");
      if (!noeof)
	 tapewriteint (tfd, 0);

      fclose (ifd);
   }

   if (verbose)
      printf ("Processed %d files\n", filecnt);

   tapewriteint (tfd, 0);
   fclose (tfd);
   return (NORMAL);
}
