/***********************************************************************
*
* obj2hex.c - Object to Intel HEX converter.
*
* Changes:
*   03/05/18   DGP   Original.
*
***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>

#include "utildef.h"
#include "hexrec.h"

#define MAXMEMLEN 64*1024
#define LOADADDRESS 0xA0
#define OBJRECLEN 80

static int outaddr = -1;
static int outlen = 0;
static int bytcnt = 0;
static char outrec[OBJRECLEN+2];

/***********************************************************************
* getnum - Get a number from the command line.
***********************************************************************/

static int
getnum (char *bp)
{
   char *cp;
   long i = 0;

   while (*bp && isspace (*bp)) bp++;
   cp = bp;
   if (*bp == '>')
   {
      i = strtol (++bp, &cp, 16);
   }
   else if (*bp == '0')
   {
      int base = 8;
      bp++;
      if (*bp == 'x' || *bp == 'X')
      {
	 bp++;
	 base = 16;
      }
      i = strtol (bp, &cp, base);
   }
   else 
   {
      i = strtol (bp, &cp, 10);
   }
   return ((int)i & 0xFFFF);
}

/***********************************************************************
* gethex - Get hex value (len bytes).
***********************************************************************/

unsigned short
gethex (char *bp, int len)
{
   int i;
   unsigned short retval, val;

   retval = 0;
   for (i = 0; i < len; i++)
   {
      val = *bp++ - '0';
      if (val > 9)
	 val -= 7;
      retval = (retval << 4) | val;
   }
   return (retval);
}

/***********************************************************************
* putrec - Put out a HEX record.
***********************************************************************/

static void
putrec (FILE *ofd)
{
   if (outlen > 0)
   {
      char *bp;
      unsigned short val, sum;
      char temp[4];

      sprintf (temp, "%02X", bytcnt);
      outrec[LENGTH_OFFSET] = temp[0];
      outrec[LENGTH_OFFSET+1] = temp[1];
      sprintf (temp, "%02X", DATA_TYPE);
      outrec[TYPE_OFFSET] = temp[0];
      outrec[TYPE_OFFSET+1] = temp[1];
      sum = 0;
      for (bp = &outrec[LENGTH_OFFSET]; *bp; bp += 2)
      {
         sum += gethex (bp, 2);
      }
      sprintf (temp, "%02X", (-sum) & 0xFF);
      outrec[DATA_OFFSET+(outlen++)] = temp[0];
      outrec[DATA_OFFSET+(outlen++)] = temp[1];
      outrec[DATA_OFFSET+(outlen++)] = 0;
      fprintf (ofd, "%s\n", outrec); 
   }
   memset (outrec, 0, OBJRECLEN+2);
   outrec[START_OFFSET] = START_CODE;
   memset (&outrec[LENGTH_OFFSET], '0', DATA_OFFSET-1);
   outlen = 0;
   bytcnt = 0;
   outaddr = -1;
}

/***********************************************************************
* putaddress - Put an address in HEX record.
***********************************************************************/

static void
putaddress (FILE *ofd, int address, unsigned short data)
{
   char temp[6];

   if (outlen > 0)
      putrec (ofd);
   sprintf (temp, "%04X", address);
   memcpy (&outrec[ADDRESS_OFFSET], temp, 4);
   outaddr = address;
}

/***********************************************************************
* putword - Puts a word into the HEX record.
***********************************************************************/

static void
putword (FILE *ofd, int address, unsigned short data)
{
   char temp[6];

   if (outlen > OBJRECLEN - DATA_OFFSET - 4)
      putrec (ofd);
   if (outaddr < 0)
      putaddress (ofd, address, data);
   sprintf (temp, "%04X", data);
   memcpy (&outrec[DATA_OFFSET+outlen], temp, 4);
   bytcnt += 2;
   outlen += 4;
}

/***********************************************************************
* puteof - Put out an EOF record.
***********************************************************************/

static void
puteof (FILE *ofd)
{
   char *bp;
   unsigned short val, sum;
   char temp[4];

   putrec (ofd);

   sprintf (temp, "%02X", EOF_TYPE);
   outrec[TYPE_OFFSET] = temp[0];
   outrec[TYPE_OFFSET+1] = temp[1];
   sum = 0;
   for (bp = &outrec[LENGTH_OFFSET]; *bp; bp += 2)
   {
      sum += gethex (bp, 2);
   }
   sprintf (temp, "%02X", (-sum) & 0xFF);
   outrec[DATA_OFFSET+(outlen++)] = temp[0];
   outrec[DATA_OFFSET+(outlen++)] = temp[1];
   outrec[DATA_OFFSET+(outlen++)] = 0;
   fprintf (ofd, "%s\n", outrec); 
}

/***********************************************************************
* binloader - Load object into specified memory area.
***********************************************************************/

static void
binloader (FILE *lfd, FILE *ofd, int loadpt)
{
   int done;
   int status;
   int binarymode;
   int wordtaglen;
   int reccount;
   int loadaddr = LOADADDRESS;
   int curraddr = LOADADDRESS;
   unsigned char inbuf[OBJRECLEN+2];

   if (loadpt >= 0)
   {
      loadaddr = loadpt;
      curraddr = loadpt;
   }

   binarymode = FALSE;
   status = fgetc (lfd);
   if (status == BINIDT_TAG)
   {
      binarymode = TRUE;
   }
   ungetc (status, lfd);

   done = FALSE;
   reccount = 0;
   wordtaglen = WORDTAGLEN;

   putrec (ofd);

   while (!done)
   {
      unsigned char *op;
      int i;

      if (binarymode)
      {
	 fread (inbuf, 1, OBJRECLEN+1, lfd);
	 if (feof(lfd))
	    return;
      }
      else
      {
	 if (fgets ((char *)inbuf, OBJRECLEN+2, lfd) == NULL)
	    return;
      }

      op = (unsigned char *)inbuf;

      if (*op == EOFSYM)
      {
	 done = TRUE;
	 break;
      }

      for (i = 0; i < OBJRECLEN; i++)
      {
	 int tmp;
	 unsigned int wdata;
	 unsigned char otag;
	 unsigned char item[16];

	 otag = *op++;
	 if ((otag == BINIDT_TAG) || binarymode)
	 {
	    wdata = (*op << 8) | *(op+1);
	 }
	 else
	 {
	    memcpy (item, op, 4);
	    item[4] = '\0';
	    sscanf ((char *)item, "%4X", &tmp);
	    wdata = tmp & 0xFFFF;
	 }
	 wdata &= 0xFFFF;

	 switch (otag)
	 {
	 case BINIDT_TAG: /* Binary IDT */
	    binarymode = TRUE;
	    wordtaglen = BINWORDTAGLEN;
	    wdata = (*op << 8) | *(op+1);
	 case IDT_TAG:
	    wdata += loadaddr;
	    if (wdata >= MAXMEMLEN)
	    {
	       fprintf (stderr, "Program too large for memory");
	       return;
	    }
	    op += IDTSIZE + wordtaglen-1;
	    break;

	 case RELORG_TAG:
	    wdata += loadaddr;
	 case ABSORG_TAG:
	    curraddr = wdata;
	    op += wordtaglen-1;
	    putaddress (ofd, curraddr, wdata);
	    break;

	 case RELDATA_TAG:
	    wdata += loadaddr;
	 case ABSDATA_TAG:
	    putword (ofd, curraddr, wdata);
	    curraddr += 2;
	    op += wordtaglen-1;
	    break;

	 case RELENTRY_TAG:
	    wdata += loadaddr;
	 case ABSENTRY_TAG:
	    op += wordtaglen-1;
	    break;

	 case PGMIDT_TAG:
	    op += IDTSIZE;
	 case NOCKSUM_TAG:
	 case CKSUM_TAG:
	    op += wordtaglen-1;
	    break;

	 case LRELGLOBAL_TAG:
	 case LABSGLOBAL_TAG:
	 case LRELEXTRN_TAG:
	 case LABSEXTRN_TAG:
	 case RELGLOBAL_TAG:
	 case ABSGLOBAL_TAG:
	 case RELEXTRN_TAG:
	 case ABSEXTRN_TAG:
	 case RELSREF_TAG:
	 case LOAD_TAG:
	 case DSEGORG_TAG:
	 case DSEGDATA_TAG:
	 case CMNGLOBAL_TAG:
	 case CMNEXTRN_TAG:
	 case COMMON_TAG:
	 case CMNORG_TAG:
	 case CMNDATA_TAG:
	 case EXTNDX_TAG:
	    fprintf (stderr, "Unlinked object not supported: tag = %c(%02X)\n",
		     otag, otag);
	    return;

	 case RELSYMBOL_TAG:
	 case ABSSYMBOL_TAG:
	    op += SYMLEN + wordtaglen-1;
	    break;

	 case LRELSYMBOL_TAG:
	 case LABSSYMBOL_TAG:
	    op += MAXSYMLEN + wordtaglen-1;
	    break;

	 case EOR_TAG:
	    i = OBJRECLEN+1;
	    break;

	 default: 
	    fprintf (stderr, "Illegal object: tag = %c(%02X)\n",
		     otag, otag);
	    return;
	 }
      }
   }

   puteof (ofd);

   return;
}

/***********************************************************************
* Main procedure
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *fd1, *fd2;
   char *infile, *outfile;
   int i;
   int loadpt;

   /*
   ** Scan off the the args.
   */

   infile = NULL;
   outfile = NULL;
   loadpt = LOADADDRESS;
   fd1 = NULL;
   fd2 = stdout;

   for (i = 1; i < argc; i++)
   {
      char *bp;

      bp = argv[i];

      if (*bp == '-')
      {
	 char *pp;

         bp++;
	 switch (*bp)
	 {
	 case 'a': /* Load address */
	    i++;
	    if (i >= argc) goto USAGE;
	    loadpt = getnum (argv[i]);
	    break;

	 case 'o': /* Output file */
	    i++;
	    if (i >= argc) goto USAGE;
	    outfile = argv[i];
	    break;

	 default:
	 USAGE:
	    fprintf (stderr, "usage: obj2hex [-options] infile.bin\n");
	    fprintf (stderr, "    -a addr        - Load address\n");
	    fprintf (stderr, "    -o outfile.hex - Output file\n");
	    return (ABORT);
	 }
      }

      else
      {
	 if (infile != NULL) goto USAGE;
         infile = argv[i];
      }

   }

   if (infile == NULL)
      goto USAGE;
   if ((fd1 = fopen (infile, "rb")) == NULL)
   {
      fprintf (stderr, "Can't open input file: %s: %s\n",
	       infile, strerror (errno));
      exit (1);
   }
   if (outfile != NULL)
   {
      if (!strcmp (infile, outfile))
      {
	 fprintf (stderr, "Input and output are the same\n");
	 fclose (fd1);
	 exit (1);
      }
      if ((fd2 = fopen (outfile, "w")) == NULL)
      {
	 fprintf (stderr, "Can't open output file: %s: %s\n",
		  outfile, strerror (errno));
	 fclose (fd1);
	 exit (1);
      }
   }

   binloader (fd1, fd2, loadpt);

   if (fd2 != stdout)
      fclose (fd2);
   fclose (fd1);
   return (0);
}
