/***********************************************************************
*
* objlink.c - Make symbolic links to object file with symbol defines.
*
* Changes:
*   01/21/14   DGP   Original.
*
***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>

#include "utildef.h"

static char filename[1024];

/***********************************************************************
* mungefilename - Escape special characters in file name.
***********************************************************************/

static void
mungefilename (char *out, char *in)
{
   for (; *in; in++)
   {
      if (*in == ' ') break;
      if (*in == '$') *out++ = '\\';
      *out++ = *in;
   }
   *out = 0;
}

/***********************************************************************
* binloader - Load object and scan for global DEFs.
***********************************************************************/

static int
binloader (FILE *lfd)
{
   int done;
   int status;
   int binarymode;
   int wordlen;
   int reccount;
   unsigned char inbuf[82];

   binarymode = FALSE;
   status = fgetc (lfd);
   if (status == BINIDT_TAG)
   {
      binarymode = TRUE;
   }
   ungetc (status, lfd);

   done = FALSE;
   reccount = 0;
   wordlen = WORDTAGLEN - 1;

   while (!done)
   {
      char *op;
      int i;

      if (binarymode)
      {
	 fread (inbuf, 1, 81, lfd);
	 if (feof(lfd))
	    return (-1);
      }
      else
      {
	 if (fgets ((char *)inbuf, 82, lfd) == NULL)
	    return (-1);
      }

#ifdef DEBUG
      printf ("record: %d\n", ++reccount);
      hexdump (stdout, inbuf, 0, 80);
#endif
      op = (char *)inbuf;
      if (*op == EOFSYM)
      {
	 return (0);
      }

      for (i = 0; i < 80; i++)
      {
	 int tmp;
	 unsigned short wdata;
	 char otag, ltag;
	 char item[MAXSYMLEN+1];
	 char linkitem[32];
	 char linkcommand[1024];

	 otag = *op++;
	 if (binarymode)
	 {
	    wdata = (*op << 8) | *(op+1);
	 }
	 else
	 {
	    memcpy (item, op, 4);
	    item[4] = '\0';
	    sscanf ((char *)item, "%4X", &tmp);
	    wdata = tmp & 0xFFFF;
	 }
	 wdata &= 0xFFFF;

#ifdef DEBUG
	 printf ("tag = %c(%02X), wdata = %04X\n", otag, otag, wdata);
#endif
	 switch (otag)
	 {
	 case BINIDT_TAG: /* Binary IDT */
	    binarymode = TRUE;
	    wordlen = BINWORDTAGLEN - 1;
	 case IDT_TAG:
	    op += wordlen;
	    strncpy (item, op, IDTSIZE);
	    item[IDTSIZE] = '\0';
	    printf ("IDT = %s, length = %04X\n", item, wdata);
	    op += IDTSIZE;
	    break;

	 case LRELEXTRN_TAG:
	 case LABSEXTRN_TAG:
	 case LRELSYMBOL_TAG:
	 case LABSSYMBOL_TAG:
	    op += (MAXSYMLEN - MAXSHORTSYMLEN);

	 case RELEXTRN_TAG:
	 case ABSEXTRN_TAG:
	 case RELSREF_TAG:
	 case LOAD_TAG:
	 case RELSYMBOL_TAG:
	 case ABSSYMBOL_TAG:
	    op += MAXSHORTSYMLEN;

	 case RELENTRY_TAG:
	 case ABSENTRY_TAG:
	 case RELDATA_TAG:
	 case ABSDATA_TAG:
	 case RELORG_TAG:
	 case ABSORG_TAG:
	 case DSEGORG_TAG:
	 case DSEGDATA_TAG:
	    op += wordlen;
	    break;

	 case RELGLOBAL_TAG:
	 case ABSGLOBAL_TAG:
	    op += wordlen;
	    strncpy (item, op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = 0;
	    mungefilename (linkitem, item);
	    op += MAXSHORTSYMLEN;
	    sprintf (linkcommand, "ln -s %s %s \n", filename, linkitem);
	    printf ("   link: %s", linkcommand);
	    system (linkcommand);
	    break;


	 case LRELGLOBAL_TAG:
	 case LABSGLOBAL_TAG:
	    op += wordlen;
	    strncpy (item, op, MAXSYMLEN);
	    item[MAXSYMLEN] = 0;
	    mungefilename (linkitem, item);
	    op += MAXSYMLEN;
	    sprintf (linkcommand, "ln -s %s %s \n", filename, linkitem);
	    printf ("   link: %s", linkcommand);
	    system (linkcommand);
	    break;

	 case CMNEXTRN_TAG:
	 case COMMON_TAG:
	    op += MAXSHORTSYMLEN;
	 case CMNORG_TAG:
	 case CMNDATA_TAG:
	 case EXTNDX_TAG:
	    op += wordlen;
	    op += wordlen;
	    break;

	 case CMNGLOBAL_TAG:
	    op += wordlen;
	    strncpy (item, op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = 0;
	    mungefilename (linkitem, item);
	    op += MAXSHORTSYMLEN;
	    op += wordlen;
	    sprintf (linkcommand, "ln -s %s %s \n", filename, linkitem);
	    printf ("   link: %s", linkcommand);
	    system (linkcommand);
	    break;

	 case PGMIDT_TAG:
	    op += IDTSIZE;
	 case NOCKSUM_TAG:
	 case CKSUM_TAG:
	    op += wordlen;
	    break; 

	 case EOR_TAG:
	    i = 81;
	    break;

	 default: 
	    fprintf (stderr,
		     "binloader: Illegal object tag: %c(%02X), last tag = %c\n",
		     otag, otag, ltag);
	    return (-22);
	 }
	 ltag = otag;
      }
   }
   return (0);
}

/***********************************************************************
* Main procedure
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *fd1;
   int i;
   int status;

   if (argc < 2)
   {
      fprintf (stderr, "usage: objlink file1.obj...\n");
      exit (1);
   }

   status = 0;
   for (i = 1; (status == 0) && (i < argc); i++)
   {
      if ((fd1 = fopen (argv[i], "rb")) == NULL)
      {
	 fprintf (stderr, "Can't open file: %s: %s\n",
		  argv[i], strerror (errno));
	 exit (1);
      }

      mungefilename (filename, argv[i]);
      status = binloader (fd1);
      fclose (fd1);
   }
   return (0);
}
