/***********************************************************************
*
* PATCHFPY - Patches a TI990 floppy disk image file.
*
* Changes:
*      12/17/08   DGP   Original.
*
***********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <errno.h>
#include <ctype.h>

#include "utiltypes.h"
#include "fd800defs.h"

#define NORMAL		0
#define ABORT		16

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#include "hexdump.h"
#include "support.h"

/***********************************************************************
* main
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *fd;
   char *fname;
   int dskloc;
   int i;
   int argndx;
   int val;
   int trk;
   int sect, offset;
   int verify;
   char buf[FPYSECLEN];
   
   fname = NULL;
   trk = -1;
   sect = -1;
   offset = -1;
   verify = -1;

   if (argc < 6) goto USAGE;
   for (argndx = 1; argndx < 6; argndx++)
   {
      switch (argndx)
      {
      case 1:
         fname = argv[argndx];
	 break;
      case 2:
         trk = getnum (argv[argndx]);
	 break;
      case 3:
         sect = getnum (argv[argndx]);
	 break;
      case 4:
	 offset = getnum (argv[argndx]);
	 break;
      case 5:
	 verify = getnum (argv[argndx]);
	 break;
      default:
      USAGE:
	 fprintf (stderr,
	"usage: patchfpy floppy.file track sector offset verify [val1 ...]\n");
	 exit (ABORT);
      }
   }
   if (fname == NULL) goto USAGE;
   if (trk < 0 || sect < 0 || offset < 0 || verify < 0) goto USAGE;

   /*
   ** Open the disk image file
   */

   if ((fd = fopen (fname, "r+b")) == NULL)
   {
      perror ("Can't open disk file");
      exit (ABORT);
   }

   if (trk >= FPYTRKDRV)
   {
      printf ("Invalid disk track specified, range [0..%d], abort\n",
	      FPYTRKDRV-1);
      fclose (fd);
      exit (ABORT);
   }
   if (sect < 1 || sect >= FPYSECTRK)
   {
      printf ("Invalid disk sector specified, range [1..%d], abort\n",
	      FPYSECTRK-1);
      fclose (fd);
      exit (ABORT);
   }
   if (offset >= FPYSECLEN)
   {
      printf ("Invalid disk offset specified, range [0..%d], abort\n",
	      FPYSECLEN-1);
      fclose (fd);
      exit (ABORT);
   }
   offset &= 0xFFFE;

   /*
   ** Calculate location
   */

   dskloc = (FPYSECTRK * FPYSECLEN * trk) + (FPYSECLEN * (sect-1));

#ifdef DEBUG
   printf ("   trk = %d, sector = %d\n",
	    trk, sect);
   printf ("   dskloc = %d\n", dskloc);
   printf ("   offset = %d(%x), verify = %d(%x)\n",
	 offset, offset, verify, verify);
#endif

   /*
   ** Go to the specified location on disk
   */

   if (fseek (fd, dskloc, SEEK_SET) < 0)
   {
      perror ("seek failed");
      fclose (fd);
      exit (ABORT);
   }

   /*
   ** Read sector
   */

   for (i = 0; i < FPYSECLEN; i++)
      buf[i] = fgetc (fd) & 0xFF;
#ifdef DEBUG
   printf ("sector:\n");
   hexdump (stdout, buf, 0, FPYSECLEN);
#endif

   /*
   ** Dump it
   */

   printf ("\ntrk %d sect %d dskloc = %d:\n", trk, sect, dskloc);
   hexdump (stdout, &buf[offset & 0xFFF0], offset & 0xFFF0, 16);

   val = ((buf[offset] & 0xFF) << 8) | (buf[offset+1] & 0xFF);
   if (val != verify)
   {
      printf ("Data does not match verification value\n");
      exit (ABORT);
   }

   i = offset;
   for (; argndx < argc; argndx++)
   {
      val = getnum (argv[argndx]);
      buf[i] = (val >> 8) & 0xFF;
      buf[i+1] = val & 0xFF;
      i += 2;
   }
   printf ("After, dskloc = %d:\n", dskloc);
   hexdump (stdout, &buf[offset & 0xFFF0], offset & 0xFFF0, 16);

   /*
   ** Go to the specified location on disk
   */

   if (fseek (fd, dskloc, SEEK_SET) < 0)
   {
      perror ("seek failed");
      fclose (fd);
      exit (ABORT);
   }

   /*
   ** Write sector
   */

   for (i = 0; i < FPYSECLEN; i++)
      if (fputc ((buf[i] & 0xFF), fd) < 0)
      {
	 perror ("Write failed");
	 exit (ABORT);
      }

   fclose (fd);

   return (NORMAL);
}
