/***********************************************************************
*
* putdisk - Puts a disk image onto the TI computer. Use in
*           conjunction with the 990 program pdisk.asm
*
* Changes:
*      07/27/15   DGP   Original
*      10/07/18   DGP   Fixed overhead.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>
#include <termios.h>
#include <time.h>

#include "support.h"
#include "simdsk.h"

#define CHWAIT 35*1000
#define MAXSECSIZE 512

static struct termios cmdtty, runtty;
static int ttifd;

/***************************************************************************
* smallsleep - sleep in U seconds
***************************************************************************/

static void
smallsleep (long Usec)
{
#if defined(UNIX)
#if defined(LINUX) || defined(SOLARIS)
#include <unistd.h>
   struct timespec req, rem;
#ifdef DEBUGCLOCK
#include <time.h>
   static clock_t starttime;
   static clock_t curtime;
   static int ticks;
   static int first = TRUE;

   if (first)
   {
      first = FALSE;
      ticks = 0;
      starttime = time (NULL) + 1;
   }
#endif

   req.tv_sec = 0;
   req.tv_nsec = Usec * 1000;

   nanosleep (&req, &rem);
#ifdef DEBUGCLOCK
   ticks++;
   curtime = time(NULL);
   if (curtime == starttime)
   {
      fprintf (stderr, "CLOCK: ticks = %d\n\r", ticks);
      starttime = curtime + 1;
      ticks = 0;
   }
#endif

#else
   struct timeval tv;

   tv.tv_sec = 0;
   tv.tv_usec = Usec;
   if (select (1, NULL, NULL, NULL, &tv) < 0)
   {
      if (errno != EINTR && errno != EAGAIN)
	 perror ("smallsleep: select failed");
   }
#endif
#endif /* UNIX */

#if defined(_WIN32)
   Sleep ((unsigned long) (Usec) / 100L);
#endif /* _WIN32 */

#if defined(OS2)
   DosSleep ((unsigned long) (Usec) / 100L);
#endif /* OS2 */
}

/*
** Terminal routines borrowed from Bub Supnik's simh
*/

static int
ttinit (FILE *ifd)
{
   ttifd = fileno(ifd);
   if (tcgetattr (ttifd, &cmdtty) < 0) return -1;	/* get old flags */
   runtty = cmdtty;
   runtty.c_lflag = runtty.c_lflag & ~(ECHO | ICANON);	/* no echo or edit */
   runtty.c_oflag = runtty.c_oflag & ~OPOST;		/* no output edit */
   runtty.c_iflag = runtty.c_iflag & ~ICRNL;		/* no cr conversion */
   runtty.c_cc[VINTR] = 0;				/* interrupt */
   runtty.c_cc[VQUIT] = 0;				/* no quit */
   runtty.c_cc[VERASE] = 0;
   runtty.c_cc[VKILL] = 0;
   runtty.c_cc[VEOF] = 0;
   runtty.c_cc[VEOL] = 0;
   runtty.c_cc[VSTART] = 0;				/* no host sync */
   runtty.c_cc[VSUSP] = 0;
   runtty.c_cc[VSTOP] = 0;
#if defined (VREPRINT)
   runtty.c_cc[VREPRINT] = 0;				/* no specials */
#endif
#if defined (VDISCARD)
   runtty.c_cc[VDISCARD] = 0;
#endif
#if defined (VWERASE)
   runtty.c_cc[VWERASE] = 0;
#endif
#if defined (VLNEXT)
   runtty.c_cc[VLNEXT] = 0;
#endif
   runtty.c_cc[VMIN] = 0;				/* no waiting */
   runtty.c_cc[VTIME] = 0;
#if defined (VDSUSP)
   runtty.c_cc[VDSUSP] = 0;
#endif
#if defined (VSTATUS)
   runtty.c_cc[VSTATUS] = 0;
#endif
   return 0;
}

static int
ttrunstate (void)
{
   runtty.c_cc[VINTR] = 0;
   if (tcsetattr (ttifd, TCSAFLUSH, &runtty) < 0) return -1;
   return 0;
}

static int
ttcmdstate (void)
{
   if (tcsetattr (ttifd, TCSAFLUSH, &cmdtty) < 0) return -1;
   return 0;
}

static int
ttclose (void)
{
   return ttcmdstate ();
}

static int
ttputchar (int out, FILE *ofd)
{
   int ttofd = fileno(ofd);
   unsigned char c;

   c = out;
   write (ttofd, &c, 1);
   smallsleep (CHWAIT);
   return 0;
}

static int
ttpoll (void)
{
   int status;
   unsigned char buf[1];

   status = read (ttifd, buf, 1);
   if (status != 1) return -1;
   else return (buf[0]);
}

/***********************************************************************
* Main program.
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *iterm;
   FILE *oterm;
   FILE *fd;
   int cyl = 0;
   int i;
   int c;
   int j, k;
   int dskloc;
   int disksize;
   int cyls, heads, sectrk, overhead, sectlen;
   int tracklen;
   unsigned short d;
   unsigned char sector[MAXSECSIZE];

   /*
   ** Check args
   */

   if (argc != 2)
   {
      fprintf (stderr, "usage: putdisk disk.file\n");
      exit (1);
   }

   /*
   ** Open the disk image file
   */

   if ((fd = fopen (argv[1], "rb")) == NULL)
   {
      perror ("Can't open disk.file");
      exit (1);
   }

   /*
   ** Read the disk geometry
   */

   if ((cyls = dskreadint (fd)) < 0)
   {
      perror ("Can't read disk geometry");
      exit (1);
   }
   if ((heads = dskreadint (fd)) < 0) 
   {
      perror ("Can't read disk geometry");
      exit (1);
   }
   heads &= 0xFFFF;
   if ((dskloc = dskreadint (fd)) < 0)
   {
      perror ("Can't read disk geometry");
      exit (1);
   }
   sectrk = dskloc & 0xFF;
   overhead = (dskloc >> 8) & 0x3FF;
   if ((sectlen = dskreadint (fd)) < 0)
   {
      perror ("Can't read disk geometry");
      exit (1);
   }

   tracklen = sectrk * (sectlen + overhead);

   for (i = 0; i < MAXDISKS; i++)
   {
      if (disks[i].cyls == cyls &&
	  disks[i].heads == heads &&
	  disks[i].sectrk == sectrk &&
	  disks[i].bytsec == sectlen &&
	  disks[i].overhead == overhead)
         break;
   }
   if (i == MAXDISKS)
   {
      printf ("*Unknown Disk model*\n");
   }
   disksize = tracklen * heads * cyls;
	       
   printf ("%s disk geometry:\n", disks[i].model);
   printf ("   cyls      = %d\n", cyls);
   printf ("   heads     = %d\n", heads);
   printf ("   sectrk    = %d\n", sectrk);
   printf ("   sectlen   = %d\n", sectlen);
   printf ("   overhead  = %d\n", overhead);
   printf ("   tracklen  = %d\n", tracklen);
   printf ("   disk size = %d bytes\n", disksize);

   /*
   ** Set up the port
   */

   if ((iterm = fopen ("/dev/ttyS0", "rb")) == NULL)
   {
      perror ("Can't open terminal");
      exit (1);
   }
   if ((oterm = fopen ("/dev/ttyS0", "w")) == NULL)
   {
      perror ("Can't open terminal");
      exit (1);
   }

   ttinit (iterm);

   /*
   ** Tell the pdisk program to go
   */

   ttputchar ('\n', oterm);

   ttrunstate();
   
   /*
   ** Send some geometry info for device check.
   */

   d = heads;
   ttputchar ((d >> 8) & 0xFF, oterm);
   ttputchar (d & 0xFF, oterm);

   d = sectrk;
   ttputchar ((d >> 8) & 0xFF, oterm);
   ttputchar (d & 0xFF, oterm);

   d = sectlen;
   ttputchar ((d >> 8) & 0xFF, oterm);
   ttputchar (d & 0xFF, oterm);

   /*
   ** Wait for ready
   */
   while ((c = ttpoll()) != 0x0D) ;

   /*
   ** Read the disk sectors and send to the pdisk program. 
   ** We send the sector length followed by the sector.
   ** If the sector is all zeros, send -1 for length to skip the send.
   */

   j =  heads * sectrk;
   for (i = 0; i < disksize; i++)
   {
      d = 0xFFFF;
      for (k = 0; k < sectlen; k++)
      {
	 if ((c = fgetc (fd)) < 0)
	 {
	    if (!feof(fd))
	       perror ("file read error");
	    goto STOP;
	 }
	 sector[k] = c & 0xFF;
	 if (c != 0) d = sectlen;
      }
      ttputchar ((d >> 8) & 0xFF, oterm);
      ttputchar (d & 0xFF, oterm);

      if (d == sectlen)
      {
	 for (k = 0; k < sectlen; k++)
	 {
	    c = sector[k];
	    ttputchar (c, oterm);
	 }
      }

      j--;
      if (j == 0)
      {
	 printf ("cyl %d\n", cyl++);
	 j = heads * sectrk;
      }

      /*
      ** Wait for go ahead
      */
      while ((c = ttpoll()) != 0x0D) ;
   }

STOP:
   ttcmdstate();
   ttclose();

   /*
   ** Clean up and quit
   */

   fclose (iterm);
   fclose (oterm);
   fclose (fd);

   return (0);
}
