/***********************************************************************
*
* RDTAPE - Read a TI990 TILINE tape image file.
*
* Changes:
*      03/31/05   DGP   Original
*      07/24/20   DGP   Fix reclen null error.
*
***********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <memory.h>

#define NORMAL		0
#define ABORT		16

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#include "hexdump.h"
#include "support.h"

/***********************************************************************
* main
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *tfd;
   FILE *ofd;
   char *bp;
   char *tapefile;
   char *outfile;
   int i;
   int done;
   int verbose;
   int binary;
   int strip;
   int chop;
   int trim;
   int reccnt;
   int reclen;
   int filenum;
   int filecnt;
   char inbuf[32767];

   ofd = stdout;

   verbose = FALSE;
   binary = FALSE;
   strip = FALSE;
   chop = FALSE;
   trim = FALSE;
   tapefile = NULL;
   outfile = NULL;
   reclen = 80;
   filenum = 1;

   /*
   ** Scan off args 
   */

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
         for (bp++; *bp; bp++) switch (*bp)
         {
	 case 'b':
	    binary = TRUE;
	    break;

	 case 'c':
	    chop = TRUE;
	    break;

	 case 'f':
	    i++;
	    filenum = atoi (argv[i]);
	    break;

	 case 'r':
	    i++;
	    reclen = atoi (argv[i]);
	    break;

	 case 's':
	    strip = TRUE;
	    break;

	 case 't':
	    trim = TRUE;
	    break;

	 case 'v':
	    verbose = TRUE;
	    break;

         default:
      USAGE:
	    printf ("usage: rdtape [-options] tape.file [out.file]\n");
            printf (" options:\n");
            printf ("    -b           - Binary output\n");
            printf ("    -c           - Chop first record CC header\n");
            printf ("    -f NN        - File number\n");
            printf ("    -r NN        - Record length\n");
            printf ("    -s           - Strip leading CRLF\n");
            printf ("    -t           - Trim tailing blanks\n");
            printf ("    -v           - Verbose output\n");
	    return (ABORT);
         }
      }
      else if (!tapefile)
      {
        tapefile = argv[i];
      }
      else if (!outfile)
      {
        outfile = argv[i];
      }
      else
      {
         goto USAGE;
      }

   }

   if (!tapefile) goto USAGE;

   /*
   ** Open the tape image file
   */

   if ((tfd = fopen (tapefile, "rb")) == NULL)
   {
      perror ("rdtape: Can't open tapefile");
      exit (ABORT);
   }

   if (outfile)
   {
      if (binary) strcpy (inbuf, "wb");
      else        strcpy (inbuf, "w");

      if ((ofd = fopen (outfile, inbuf)) == NULL)
      {
	 perror ("rdtape: Can't open outfile");
	 exit (ABORT);
      }
   }

   if (verbose)
   {
      fprintf (stderr, "rdtape: Reading file %s\n", tapefile);
      fprintf (stderr, "        Writing file %s\n",
				 outfile ? outfile : "stdout");
      fprintf (stderr, "        FileNum = %d\n", filenum);
      fprintf (stderr, "        Reclen = %d\n", reclen);
      fprintf (stderr, "        Binary = %s\n", binary ? "TRUE" : "FALSE");
      fprintf (stderr, "        Chop   = %s\n", chop ? "TRUE" : "FALSE");
      fprintf (stderr, "        Strip  = %s\n", strip ? "TRUE" : "FALSE");
      fprintf (stderr, "        Trim   = %s\n", trim ? "TRUE" : "FALSE");
   }

   reccnt = 0;
   filecnt = 1;
   done = FALSE;
   while (!done)
   {
      i = tapereadint (tfd);
      if (i <= 0)
      {
	 if (filecnt == filenum)
	    break;
	 if (verbose)
	 {
	    fprintf (stderr, "file %d EOF\n", filecnt);
	 }
	 filecnt++;
	 continue;
      }
      fread (inbuf, i, 1, tfd);
      if (verbose && !chop)
      {
         fprintf (stderr, "read file %d record %05d, length = %05d\n",
	 	  filecnt, ++reccnt, i);
	 HEXDUMP (stderr, inbuf, i, 0);
      }
      tapereadint (tfd);
      if (filecnt < filenum)
         continue;
      if (i < reclen)
      {
	 for (; i < reclen; i++)
	    inbuf[i] = ' ';
      }
      if (!binary)
	 inbuf[reclen] = '\0';

      if (!chop)
      {
	 if (binary)
	 {
	    fwrite (inbuf, i, 1, ofd);
	 }
	 else
	 {
	    if (strip)
	    {
	       if (inbuf[0] == '\r' && inbuf[1] == '\n')
		  memmove (inbuf, &inbuf[2], i);
	    }
	    if (trim)
	    {
	       for (i = reclen - 1; i > 1 && inbuf[i] == ' '; i--)
	          inbuf[i] = '\0';
	    }
	    strcat (inbuf, "\n");
	    fputs (inbuf, ofd);
	 }
      }
      else
         chop = FALSE;
   }

   fclose (tfd);
   if (ofd != stdout) fclose (ofd);
   return (NORMAL);
}
