/***********************************************************************
*
* TCOMP - Compare TI-990 disk image with tape.
*
* Changes:
*   10/15/18   DGP   Original
*
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <memory.h>

#include "tapeio.h"
#include "diskio.h"
#include "hexdump.h"

#define BLOCKSIZE 512
#define SECSBLOCK 40

static char buffer[BLOCKSIZE*SECSBLOCK];
static char dskbuf[BLOCKSIZE];

extern void lights(int);

static int
getnum (char *buf)
{
   unsigned char *bp = (unsigned char *)buf;
   return (bp[0] << 8 | bp[1]);
}

int
main (int argc, char **argv)
{
   long outsecs;
   int done;
   int i, j, k;
   int cyls, heads, sects, sectlen;
   int cyl, head, sect, len;
   int maxcyl;
   int verbose;
   int debug;
   int ch;

   verbose = 0;
   debug = 0;
   maxcyl = -1;
   for (i = 1; i < argc; i++)
   {
      char *ap = argv[i];
      if (*ap == '-')
      {
         ap++;
         if (*ap == 'd')
	    verbose = debug = 1;
         else if (*ap == 'v')
	    verbose = 1;
         else
	 {
	 USAGE:
	    printf ("usage: tcomp [-v]\n");
	    return (1);
	 }
      }
      else
         goto USAGE;
   }

   if (diskinit (0) < 0)
   {
      printf ("tcomp: can't initialize disk\n");
      return (1);
   }
   
   if (diskgeometry (0, &cyls, &heads, &sects, &sectlen) < 0)
   {
      printf ("tcomp: can't get disk geometry\n");
      return (1);
   }

   if (verbose)
   {
      printf ("tcomp: Disk Geometry:\n");
      printf ("   cyls = %d, heads = %d, sects = %d, sectlen = %d\n",
	      cyls, heads, sects, sectlen);
   }

   if (taperewind (0) < 0)
   {
      printf ("tcomp: can't rewind tape\n");
      return (1);
   }

   if (taperead (0, buffer, BLOCKSIZE) < 0)
   {
      printf ("tcomp: can't read geometry from tape\n");
      return (1);
   }

   cyl = getnum (&buffer[0]);
   head = getnum (&buffer[2]);
   sect = getnum (&buffer[4]);
   len = getnum (&buffer[6]);
   if (verbose)
   {
      printf ("tcomp: Disk Geometry on tape:\n");
      printf ("   cyls = %d, heads = %d, sects = %d, sectlen = %d\n",
	      cyl, head, sect, len);
   }

   if (cyl != cyls)
   {
      printf ("tcomp: geometry mismatch: cyls = %d != %d\n",
	       cyls, cyl);
      return (1);
   }
   if (head != heads)
   {
      printf ("tcomp: geometry mismatch: heads = %d != %d\n",
	       heads, head);
      return (1);
   }
   if (sect != sects)
   {
      printf ("tcomp: geometry mismatch: sects = %d != %d\n",
	       sects, sect);
      return (1);
   }
   if (len != sectlen)
   {
      printf ("tcomp: geometry mismatch: sectlen = %d != %d\n",
	       sectlen, len);
      return (1);
   }

   printf ("tcomp: press enter to start: ");
   fflush (stdout);
   ch = getchar();

   if (verbose)
      printf ("tcomp: compare tape to disk\n");

   done = 0;
   outsecs = 0;
   while (!done)
   {
      char *bp;

      if ((len = taperead (0, buffer, sizeof(buffer))) < 0)
      {
	 printf ("tcomp: can't read data from tape\n");
	 return (1);
      }
      if (len == 0)
      {
	 done = 1;
	 break;
      }

      bp = buffer;
      for (; len > 0; len -= BLOCKSIZE)
      {
	 cyl = getnum (bp);
	 head = getnum (bp+2);
	 sect = getnum (bp+4);
	 lights (cyl << 5 | head);
	 if (debug)
	 {
	    printf ("   write: cyl = %d, head = %d, sect = %d\n",
		    cyl, head, sect);
#ifdef DEBUG
	    hexdump (stdout, bp+8, 0, sectlen);
#endif
	 }
	 memset (dskbuf, 0xFF, sectlen);
	 if (diskread (0, cyl, head, sect, dskbuf, sectlen) < 0)
	 {
	    printf ("tcomp: can't read data from disk\n");
	    return (1);
	 }
	 if (memcmp (dskbuf, bp+8, sectlen) != 0)
	 {
	    printf ("compare error: cyl = %d, head = %d, sect = %d, len = %d\n",
		    cyl, head, sect, sectlen);
	    printf ("disk buffer:\n");        
	    hexdump (stdout, dskbuf, 0, sectlen);
	    printf ("tape buffer:\n");        
	    hexdump (stdout, bp+8, 0, sectlen);
	    return (0);
	 }
	 bp += BLOCKSIZE;
	 outsecs++;
      }
   }

   taperewind (0);
   if (verbose)
      printf ("tcomp: DONE: %ld sectors compared\n", outsecs);
   return (0);
}
