/***********************************************************************
*
* TCOMP - Compare TI-990 disk image with tape.
*
* Changes:
*   10/15/18   DGP   Original
*
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <memory.h>

#include "tapeio.h"
#include "diskio.h"
#include "hexdump.h"

#define BLOCKSIZE 512
#define SECSBLOCK 40

static int cyls, heads, sects, sectlen;
static int verbose;
static int debug;
static char buffer[BLOCKSIZE*SECSBLOCK];
static char tapbuf[BLOCKSIZE];
static char dskbuf[BLOCKSIZE];

extern void lights(int);


static int
getnum (char *buf)
{
   unsigned char *bp = (unsigned char *)buf;
   return (bp[0] << 8 | bp[1]);
}

static int
tapeblock (int *cyl, int *head, int *sect, int sectlen, char *tbuf, int tlen)
{
   static char *bp = 0;
   static int len = 0;
   int lcyl, lhead, lsect;
   int ncyl, nhead, nsect;

   if (len == 0)
   {
      if ((len = taperead (0, buffer, sizeof(buffer))) <= 0)
         return (len);
      bp = buffer;
   }

   *cyl = lcyl = getnum (bp);
   *head = lhead = getnum (bp+2);
   *sect = lsect = getnum (bp+4);
   lights (lcyl << 5 | lhead);
   if (debug)
   {
      printf ("   read: cyl = %d, head = %d, sect = %d\n",
	      lcyl, lhead, lsect);
#ifdef DEBUG
      hexdump (stdout, bp+8, 0, sectlen);
#endif
   }
   memcpy (tbuf, bp+8, sectlen);
   bp += BLOCKSIZE;
   len -= BLOCKSIZE;

   if (len == 0)
   {
      if ((len = taperead (0, buffer, sizeof(buffer))) <= 0)
         return (len);
      bp = buffer;
   }

   lsect++;
   if (lsect == sects)
   {
      lsect = 0;
      lhead++;
      if (lhead == heads)
      {
	 lhead = 0;
	 lcyl++;
	 if (lcyl == cyls)
	    lcyl = 0;
      }
   }

   ncyl = getnum (bp);
   nhead = getnum (bp+2);
   nsect = getnum (bp+4);
   if (lcyl == ncyl && lhead == nhead && lsect == nsect)
   {
      memcpy (&tbuf[sectlen], bp+8, sectlen);
      bp += BLOCKSIZE;
      len -= BLOCKSIZE;
   }
   else
   {
      memset (&tbuf[sectlen], 0, sectlen);
   }
   return (BLOCKSIZE);
}

int
main (int argc, char **argv)
{
   long outblks;
   int done;
   int i, j, k;
   int cyl, head, sect, len;
   int maxcyl;
   int ch;

   verbose = 0;
   debug = 0;
   maxcyl = -1;
   for (i = 1; i < argc; i++)
   {
      char *ap = argv[i];
      if (*ap == '-')
      {
         ap++;
         if (*ap == 'd')
	    verbose = debug = 1;
         else if (*ap == 'v')
	    verbose = 1;
         else
	 {
	 USAGE:
	    printf ("usage: tcomp2 [-v]\n");
	    return (1);
	 }
      }
      else
         goto USAGE;
   }

   if (diskinit (0) < 0)
   {
      printf ("tcomp2: can't initialize disk\n");
      return (1);
   }
   
   if (diskgeometry (0, &cyls, &heads, &sects, &sectlen) < 0)
   {
      printf ("tcomp2: can't get disk geometry\n");
      return (1);
   }

   if (verbose)
   {
      printf ("tcomp2: Disk Geometry:\n");
      printf ("   cyls = %d, heads = %d, sects = %d, sectlen = %d\n",
	      cyls, heads, sects, sectlen);
   }

   if (taperewind (0) < 0)
   {
      printf ("tcomp2: can't rewind tape\n");
      return (1);
   }

   if (taperead (0, buffer, BLOCKSIZE) < 0)
   {
      printf ("tcomp2: can't read geometry from tape\n");
      return (1);
   }

   cyl = getnum (&buffer[0]);
   head = getnum (&buffer[2]);
   sect = getnum (&buffer[4]);
   len = getnum (&buffer[6]);
   if (verbose)
   {
      printf ("tcomp2: Disk Geometry on tape:\n");
      printf ("   cyls = %d, heads = %d, sects = %d, sectlen = %d\n",
	      cyl, head, sect, len);
   }

   if (cyl != cyls)
   {
      printf ("tcomp2: geometry mismatch: cyls = %d != %d\n",
	       cyls, cyl);
      return (1);
   }
   if (head != heads)
   {
      printf ("tcomp2: geometry mismatch: heads = %d != %d\n",
	       heads, head);
      return (1);
   }
   if (sect != sects)
   {
      printf ("tcomp2: geometry mismatch: sects = %d != %d\n",
	       sects, sect);
      return (1);
   }
   if (len != sectlen)
   {
      printf ("tcomp2: geometry mismatch: sectlen = %d != %d\n",
	       sectlen, len);
      return (1);
   }

   printf ("tcomp2: press enter to start: ");
   fflush (stdout);
   ch = getchar();

   if (verbose)
      printf ("tcomp2: compare tape to disk\n");

   done = 0;
   outblks = 0;
   while (!done)
   {

      if ((len = tapeblock (&cyl, &head, &sect, sectlen, tapbuf, BLOCKSIZE)) < 0)
      {
	 printf ("tcomp2: can't read data from tape\n");
	 return (1);
      }
      if (len == 0)
      {
	 done = 1;
	 break;
      }

      memset (dskbuf, 0xFF, BLOCKSIZE);
      if (diskread (0, cyl, head, sect, dskbuf, BLOCKSIZE) < 0)
      {
	 printf ("tcomp2: can't read data from disk\n");
	 return (1);
      }
      if (memcmp (dskbuf, tapbuf, BLOCKSIZE) != 0)
      {
	 printf ("compare error: cyl = %d, head = %d, sect = %d, len = %d\n",
		 cyl, head, sect, BLOCKSIZE);
	 printf ("disk buffer:\n");        
	 hexdump (stdout, dskbuf, 0, BLOCKSIZE);
	 printf ("tape buffer:\n");        
	 hexdump (stdout, tapbuf, 0, BLOCKSIZE);
	 return (0);
      }
      outblks++;
   }

   taperewind (0);
   if (verbose)
      printf ("tcomp2: DONE: %ld blocks compared\n", outblks);
   return (0);
}
