/*#define DEBUG*/
/*#define DEBUGMBUF*/
#include <stdio.h>
#include <fcntl.h>
#include <sgtty.h>

#include "net.h"
#include "mbuf.h"
#include "ip.h"
#include "ifcb.h"

#define SL0_DEV "/dev/tty3"
#define MAXCOUNT 2

/* SLIP special character codes */
#define END             0300    /* indicates end of packet */
#define ESC             0333    /* indicates byte stuffing */
#define ESC_END         0334    /* ESC ESC_END means END data byte */
#define ESC_ESC         0335    /* ESC ESC_ESC means ESC data byte */

static int slfd;

/* Write one character (byte) to the TTY link. */
static int
send_char (c)
   int c;
{
   unsigned char buf[2];

   buf[0] = c;
   write (slfd, buf, 1);
   return (0);
}


/*
* SL0_WRITE: write a packet of length "len" from a chain of mbufs.
*/

static void
sl0_write (ip, m, len)
   register struct ifcb *ip;
   struct mbuf *m;
   int len;
{
   register struct mbuf *n;
   register unsigned char *p;
   register int sent;

   /*
   * send an initial END character to flush out any data that may
   * have accumulated in the receiver due to line noise
   */
   send_char (END);

   sent = 0;
   for (n = m; n; n = n->m_next) {
      p = (unsigned char *)((unsigned int)n + n->m_off);
      len = n->m_len;

      /*
      * for each byte in the packet, send the appropriate character sequence
      */
      while (len--)
      {
	 switch (*p)
	 {

	 /* if it's the same code as an END character, we send a
	  * special two character code so as not to make the
	  * receiver think we sent an END
	  */
	 case END:
	    send_char (ESC);
	    send_char (ESC_END);
	    break;

	 /* if it's the same code as an ESC character,
	  * we send a special two character code so as not
	  * to make the receiver think we sent an ESC
	  */
	 case ESC:
	    send_char (ESC);
	    send_char (ESC_ESC);
	    break;

	 /* otherwise, we just send the character
	  */
	 default:
	    send_char (*p);
	 }

	 p++;

	 if (sent++ >= ip->if_mtu) {
	    send_char (END);
	    ip->if_opkts++;
	    sent = 0;
	 }
      }
  }

   /* tell the receiver that we're done sending the packet */
   if (sent) {
      send_char (END);
      ip->if_opkts++;
   }
}


/* Read one character (byte) from the TTY link. */
static int
recv_char ()
{
   unsigned char buf[2];

   read (slfd, buf, 1);
   return (buf[0]);
}

/*
* SL0_READ: reads a packet into allocated mbufs.
*      If more than ip->if_mtu bytes are received, the packet will
*      be truncated.
*      Returns the first allocated mbuf.
*/
static struct mbuf *
sl0_read (ip)
   register struct ifcb *ip;
{
   struct mbuf *m;
   struct mbuf *q;
   register struct mbuf *n;
   register unsigned char *p;
   unsigned char c;
   int received = 0;
   int i;

#ifdef DEBUG
   printf ("sl0_read: entered\n");
#endif

   /* sit in a loop reading bytes until we put together
    * a whole packet.
    * Make sure not to copy them into the packet if we
    * run out of room.
    */
   if ((m = m_get (0)) == NULL)
   {
#ifdef DEBUGMBUF
      printf ("sl0_read: NO MBUFs-1\n");
#endif
      return (NULL);
   }

   n = m;
   p = (unsigned char *)((unsigned int)n + n->m_off);
   while (1)
   {
      /* get a character to process */
      c = recv_char ();

      /* handle bytestuffing if necessary */
      switch (c)
      {

      /* if it's an END character then we're done with the packet */
      case END:
	 /* a minor optimization: if there is no
	  * data in the packet, ignore it. This is
	  * meant to avoid bothering IP with all
	  * the empty packets generated by the
	  * duplicate END characters which are in
	  * turn sent to try to detect line noise.
	  */
	 if (received) {
	    ip->if_ipkts++;
	    if (ip->if_inq_hd != NULL)
	       ip->if_inq_tl->m_act = m;
	    else
	       ip->if_inq_hd = m;
	    ip->if_inq_tl = m;
#ifdef DEBUG
	    for (n = m; n; n = n->m_next) {
	       int ii;
	       printf (
	        "mbuf >%x: m_next = >%x, m_act = >%x, m_len = %d, m_off = %d\n",
		       n, n->m_next, n->m_act, n->m_len, n->m_off);
#if 0
	       p = (unsigned char *)((unsigned int)n + n->m_off);
	       ii = 0;
	       for (i = 0; i < n->m_len; i++) {
		  printf ("  >%x", *p++);
		  if (ii++ > 6) {
		     ii = 0;
		     printf ("\n");
		  }
	       }
	       if (ii) printf ("\n");
#endif
	    }
#endif
	    return m;
	 }
	 else
	    break;

      /* if it's the same code as an ESC character, wait
       * and get another character and then figure out
       * what to store in the packet based on that.
       */
      case ESC:
	 c = recv_char ();

	 /* if "c" is not one of these two, then we
	  * have a protocol violation.  The best bet
	  * seems to be to leave the byte alone and
	  * just stuff it into the packet
	  */
	 if (c == ESC_END)
	    c = END;
	 else if (c == ESC_ESC)
	    c = ESC;

      /* here we fall into the default handler and let
       * it store the character for us
       */
      default:
	 if (received >= ip->if_mtu) {
	       printf ("sl0_read: received %d > mtu %d\n",
	       		received, ip->if_mtu);
	 } else {
	    if (n->m_len >= MLEN) {
	       if ((q = m_get (0)) == NULL)
	       {
#ifdef DEBUGMBUF
		  printf ("sl0_read: NO MBUFs-2\n");
#endif
		  return m;
	       }
	       n->m_next = q;
	       n = q;
	       p = (unsigned char *)((unsigned int)n + n->m_off);
#ifdef DEBUG
	       printf ("sl0_read: new mbuf = 0x%x\n", q);
#endif
	    }
	    p[n->m_len++] = c;
	    received++;
	 }
      }
   }
}

int
sl0_send (m, ip, addr, len, link)
   register struct mbuf *m;
   register struct ifcb *ip;
   struct socket *addr;
   register len;
   int link;
{
#ifdef DEBUG
   register struct mbuf *n;
   unsigned char *p;
   int i;

   printf ("sl0_send: entered\n");
   for (n = m; n; n = n->m_next) {
      printf ("mbuf >%x: m_next = >%x, m_act = >%x, m_len = %d, m_off = %d\n",
	      n, n->m_next, n->m_act, n->m_len, n->m_off);
#if 0
      p = (unsigned char *)((int)n + n->m_off);
      for (i = 0; i < n->m_len; i++)
	 printf ("  >%x\n", *p++);
#endif
   }
#endif

   m->m_act = NULL;
   sl0_write (ip, m, len);

#ifdef LOOPBACK
   if (ip->if_inq_hd != NULL)
      ip->if_inq_tl->m_act = m;
   else
      ip->if_inq_hd = m;
   ip->if_inq_tl = m;
#else
   m_freem (m);
#endif

   return (TRUE);
}

int
sl0_rcv (ip)
   register struct ifcb *ip;
{
   register struct mbuf *m;

#ifdef DEBUG
   printf ("sl0_rcv: entered\n");
#endif

   while ((m = ip->if_inq_hd) != NULL) {
      ip->if_inq_hd = m->m_act;
      m->m_act = NULL;
      ip_input (m, ip);
   }

   return 0;
}

int
sl0_raw ()
{
   return 0;
}

int
sl0_init (ip)
   register struct ifcb *ip;
{
   struct sgttyb ttyb;

   if ((slfd = open (SL0_DEV, O_RDWR)) < 0)
      return -1;

   ttyb.sg_ispeed = ttyb.sg_ospeed = B9600;
   ttyb.sg_erase = ttyb.sg_kill = 0;
   ttyb.sg_flags = (RAW | CBREAK | SLIP);
   if (stty (slfd, &ttyb) < 0) {
      close (slfd);
      return -1;
   }

   ip->if_avail = 1;

   return 0;
}

int
sl0_out ()
{
   return 0;
}

int
sl0_poll (ip)
   register struct ifcb *ip;
{
   struct sgttyb ttyb;

   if (gtty (slfd, &ttyb) < 0)
      return (-1);

   if (ttyb.sg_flags & DAVAIL) {
      if (sl0_read (ip))
	 return (1);
   }
   return 0;
}
