/*
 * 911 VDT driver - runs as a "glass" tty
 */

#include "param.h"
#include "conf.h"
#include "tty.h"
#include "user.h"
#include "proc.h"

#define NACA 2

/* Bit offsets on a 911 vdt board */

/* SETB word 0 */
#define	V911HILO	7
#define	V911WSTB	8
#define V911TEST	9
#define	V911CMOV	10
#define	V911BLNK	11
#define	V911KENB	12
#define	V911DUAL	13
#define	V911DENB 	14
#define V911WSEL	15
/* SETB word 1 */
#define V911DCSR	12
#define V911KACK	13
#define V911BEEP	14

/* TB word 0 */
#define V911KRDY	15
/* TB word 1 */
#define V911CMSB	10
#define V911KMSB	11
#define V911DTR		12
#define V911PST		13
#define V911KPAR	14

#define V911SIZE	1920

/*
** 911 Function button definitions, most are ignored.
*/

#ifdef DELETE
#undef DELETE
#endif

#define EFIELD 0x80
#define EINPUT 0x81
#define HOME   0x82
#define HTAB   0x83
#define DELETE 0x84
#define SKIP   0x85
#define INSERT 0x86
#define LEFTF  0x87
#define LEFT   0x88
#define UP     0x89
#define RIGHT  0x8A
#define DOWN   0x8B
#define RIGHTF 0x8C
#define XF1    0x8D
#define XF2    0x8E
#define XF3    0x8F
#define F1     0x92
#define F2     0x93
#define F3     0x94
#define F4     0x95
#define F5     0x96
#define F6     0x97
#define F7     0x98
#define F8     0x99
#define PRINT  0x9A
#define CMD    0x9B
#define GOLD   0x9C

/* 911 VDT CRU addresses */
extern int v911addrs[NACA];
extern int v911genint[NACA];

struct control {
	int init;
	int pause;
	struct tty aca;
};

static struct control v911dat[NACA];

extern void v911scroll ();
extern void v911clear ();

/* Open the device: set initial flags and stat, clear screen and enable
 * keyboard interrupt.
 */
void
v911open (dev, flag)
{
	register int minor, *addr;
	register struct tty *tp;
	extern void v911start();

	minor = MINOR(dev);
	if (minor >= NACA) {
		u.u_error = ENXIO;
		return;
	}

	addr = (int *)v911addrs[minor];
	if (addr == 0xFFFF || stcr16 (addr) == 0xFFFF) {
		u.u_error = ENXIO;
		return;
	}

	tp = &v911dat[minor].aca;
	if (u.u_procp->p_pid > 1) {
		u.u_procp->p_ttyp = tp;
		tp->t_dev = dev;
	}

	tp->t_addr = addr;
	tp->t_start = v911start;
	if ((tp->t_state & ISOPEN) == 0) {
		tp->t_state = ISOPEN | CARR_ON;
		tp->t_flags = XTABS | ECHO | CRMOD;
		tp->t_speeds = (B9600 << 8) | B9600;
		tp->t_erase = CERASE;
		tp->t_kill  = CKILL;
	}

	v911dat[minor].pause = 1;
	if (v911dat[minor].init == 0) {
		sbz (addr, V911WSEL);
		sbz (addr, V911TEST);
		sbo (addr, V911BLNK);
		sbo (addr, V911DUAL);
		v911clear (addr);
		v911dat[minor].init = 1;
	}
	sbz (addr, V911WSEL);
	sbo (addr, V911KENB);
	sbo (addr, V911WSEL);
	ldcr (addr, 11, V911SIZE-80);
	v911dat[minor].pause = 0;
}

/* Close the device: wait for the output queue to flush.
 * And disable keyboard interrupts.
 */
void
v911close (dev, flag)
{
	register struct tty *tp;
	int *addr;

	tp = &v911dat[MINOR(dev)].aca;
	addr = tp->t_addr;
	wflushtty (tp);
	sbz (addr, V911WSEL);
	sbz (addr, V911KENB);
	tp->t_state = 0;
}

/* Start the transmitting. 
 * If the output queue is empty, disable the interrupt until there is
 * something to send. Otherwise, place the next character in the xmit buffer.
 */
void
v911start (tp)
	register struct tty *tp;
{
	register int c, minor, *addr;

	addr = tp->t_addr;
	minor = MINOR(tp->t_dev);

	/* Tell the clock handler to generate fake xmit interrupts. */
	v911genint[minor] = 1;

	/* If scrolling or clearing screen, wait a bit. */
	if (v911dat[minor].pause) return;

	while ((c = getc (&tp->t_outq)) >= 0) {
		v911dat[minor].pause = 1;

		if (c == '\n') {
			v911scroll (addr);
			sbo (addr, V911WSEL);
			ldcr (addr, 11, V911SIZE-80);
		}
		else if (c == tp->t_erase) {
			sbz (addr, V911WSEL);
			sbo (addr, V911CMOV);
		}
		else if (c >= ' ') {
			sbz (addr, V911WSEL);
			ldcr8 (addr, c | 0x80);
			sbz (addr, V911WSTB);
			sbz (addr, V911CMOV);
		}
		v911dat[minor].pause = 0;
	}
}


/* Handle keyboard interrupt.
 * Read the character from the device and reset and reenable the interrupt
 * signal. Place the character on the input queue (if the queue is full, the
 * character gets dropped).
 */
void
v911rint (xea)
{
	register int c, minor, *addr;
	register struct tty *tp;

	minor = xea & 0x3f;
	tp = &v911dat[minor].aca;
	addr = tp->t_addr;
	sbo (addr, V911WSEL);
	sbz (addr, V911KACK);
	sbz (addr, V911WSEL);
	c = stcr8 (addr + 8) & 0x7f;
	sbo (addr, V911WSEL);
	if (tb (addr, V911KMSB)) c |= 0x80;
	if (c == LEFT || c == DELETE) c = tp->t_erase;
	else if (c == EINPUT || c == EFIELD || c == LEFTF) c = tp->t_kill;
	else if (c == CMD) c = CINTR;
	else if (c == GOLD) c = CQUIT;
	if (c <= 0x7f) {
		ttyinput (c, tp);
	}
}

/* Handle transmitter buffer empty interrupt; Generated by clock interrupt.
 * Start/stop sending as needed. If the process was suspended on a full queue
 * wake it up again once on the low water mark or on an empty queue.
 */
void
v911xint (xea)
{
	register int minor;
	register struct tty *tp;

	minor = xea & 0x3f;
	tp = &v911dat[minor].aca;
	v911start (tp);
	if (tp->t_outq.c_cc == 0 || tp->t_outq.c_cc == TTLOWAT) {
		v911genint[minor] = 0;
		wakeup (&tp->t_outq);
	}
}

/*
 * COMMON routines.
 */

void
v911read (dev)
{
	ttread (&v911dat[MINOR(dev)].aca);
}

void
v911write (dev)
{
	ttwrite (&v911dat[MINOR(dev)].aca);
}

void
v911sgtty (dev, v)
	int *v;
{
	register struct tty *tp;

	tp = &v911dat[MINOR(dev)].aca;
	ttystty (tp, v);
}

